'use server'

import { sendDiscordNotification } from '@/lib/discord'
import { auth } from '@/auth'
import { db } from '@/lib/db'
import { users, creatorSocials } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { revalidatePath } from 'next/cache'

export async function suspendCreator(creatorId: string) {
    const session = await auth()
    if (!session || session.user.role !== 'admin') {
        return { error: 'Unauthorized' }
    }

    try {
        // Toggle suspension (mocked as we don't have a suspended status, let's use isApproved = 0 for now or add a status)
        await db.update(users).set({ isApproved: 0 }).where(eq(users.id, creatorId))

        await sendDiscordNotification(
            '🚫 User Suspended',
            `Admin ${session.user.email} suspended user ${creatorId}`,
            'warning',
            [
                { name: 'Admin', value: session.user.email || 'Unknown', inline: true },
                { name: 'User ID', value: creatorId, inline: true }
            ]
        )

        revalidatePath('/ops/creators')
        return { success: true }
    } catch (error) {
        console.error('Failed to suspend creator:', error)
        return { error: 'Failed to suspend creator' }
    }
}

export async function deleteCreator(creatorId: string) {
    const session = await auth()
    if (!session || session.user.role !== 'admin') {
        return { error: 'Unauthorized' }
    }

    try {
        await db.delete(users).where(eq(users.id, creatorId))

        await sendDiscordNotification(
            '🗑️ User Deleted',
            `Admin ${session.user.email} deleted user ${creatorId}`,
            'error',
            [
                { name: 'Admin', value: session.user.email || 'Unknown', inline: true },
                { name: 'User ID', value: creatorId, inline: true }
            ]
        )

        revalidatePath('/ops/creators')
        return { success: true }
    } catch (error) {
        console.error('Failed to delete creator:', error)
        return { error: 'Failed to delete creator' }
    }
}

export async function verifySocial(socialId: string) {
    const session = await auth()
    if (!session || session.user.role !== 'admin') {
        return { error: 'Unauthorized' }
    }

    try {
        await db.update(creatorSocials)
            .set({ verificationStatus: 'approved', rejectionReason: null })
            .where(eq(creatorSocials.id, socialId))

        revalidatePath('/ops/creators')
        revalidatePath('/ops/verifications')
        return { success: true }
    } catch (error) {
        console.error('Failed to verify social:', error)
        return { error: 'Failed to verify social' }
    }
}

export async function rejectSocial(socialId: string, reason: string) {
    const session = await auth()
    if (!session || session.user.role !== 'admin') {
        return { error: 'Unauthorized' }
    }

    try {
        await db.update(creatorSocials)
            .set({ verificationStatus: 'rejected', rejectionReason: reason })
            .where(eq(creatorSocials.id, socialId))

        revalidatePath('/ops/creators')
        revalidatePath('/ops/verifications')
        return { success: true }
    } catch (error) {
        console.error('Failed to reject social:', error)
        return { error: 'Failed to reject social' }
    }
}
