'use server'

import { auth } from '@/auth'
import { db } from '@/lib/db'
import { users, campaigns } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { revalidatePath } from 'next/cache'
import { sendDiscordNotification } from '@/lib/discord'

export async function approveUser(userId: string) {
    const session = await auth()
    if (!session || session.user.role !== 'admin') {
        return { error: 'Unauthorized' }
    }

    try {
        const [user] = await db.update(users)
            .set({ isApproved: 1 })
            .where(eq(users.id, userId))
            .returning()

        if (!user) return { error: 'User not found' }

        await sendDiscordNotification(
            '✅ Utilisateur Approuvé',
            `L'admin **${session.user.email}** a approuvé **${user.email}** (${user.role}).`,
            'success'
        )

        revalidatePath('/ops/moderation')
        revalidatePath('/ops/creators')
        revalidatePath('/ops/studios')
        return { success: true }
    } catch (error) {
        console.error('Error approving user:', error)
        return { error: 'Failed to approve user' }
    }
}

export async function rejectUser(userId: string) {
    const session = await auth()
    if (!session || session.user.role !== 'admin') {
        return { error: 'Unauthorized' }
    }

    try {
        // For rejection, we might want to delete the user or just mark as rejected?
        // Usually "rejecting" a request means deleting the account or setting a status.
        // Let's delete for now as it's a "request".

        const user = await db.query.users.findFirst({
            where: eq(users.id, userId)
        })

        if (!user) return { error: 'User not found' }

        await db.delete(users).where(eq(users.id, userId))

        await sendDiscordNotification(
            '❌ Utilisateur Rejeté',
            `L'admin **${session.user.email}** a rejeté (supprimé) **${user.email}** (${user.role}).`,
            'warning'
        )

        revalidatePath('/ops/moderation')
        revalidatePath('/ops/creators')
        revalidatePath('/ops/studios')
        return { success: true }
    } catch (error) {
        console.error('Error rejecting user:', error)
        return { error: 'Failed to reject user' }
    }
}

export async function approveCampaign(campaignId: string) {
    const session = await auth()
    if (!session || session.user.role !== 'admin') {
        return { error: 'Unauthorized' }
    }

    try {
        const [campaign] = await db.update(campaigns)
            .set({ status: 'active' })
            .where(eq(campaigns.id, campaignId))
            .returning()

        if (!campaign) return { error: 'Campaign not found' }

        await sendDiscordNotification(
            '✅ Campagne Approuvée',
            `L'admin **${session.user.email}** a approuvé la campagne **${campaign.title}**.`,
            'success'
        )

        revalidatePath('/ops/moderation')
        return { success: true }
    } catch (error) {
        console.error('Error approving campaign:', error)
        return { error: 'Failed to approve campaign' }
    }
}

export async function rejectCampaign(campaignId: string) {
    const session = await auth()
    if (!session || session.user.role !== 'admin') {
        return { error: 'Unauthorized' }
    }

    try {
        const [campaign] = await db.update(campaigns)
            .set({ status: 'rejected' })
            .where(eq(campaigns.id, campaignId))
            .returning()

        if (!campaign) return { error: 'Campaign not found' }

        await sendDiscordNotification(
            '❌ Campagne Rejetée',
            `L'admin **${session.user.email}** a rejeté la campagne **${campaign.title}**.`,
            'warning'
        )

        revalidatePath('/ops/moderation')
        return { success: true }
    } catch (error) {
        console.error('Error rejecting campaign:', error)
        return { error: 'Failed to reject campaign' }
    }
}
