'use server'

import { sendDiscordNotification } from '@/lib/discord'
import { auth } from '@/auth'
import { db } from '@/lib/db'
import { globalSettings } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { revalidatePath } from 'next/cache'

export async function updateAdminSettings(data: any) {
    const session = await auth()

    if (!session || session.user.role !== 'admin') {
        return { error: 'Unauthorized' }
    }

    try {
        const existingSettings = await db.query.globalSettings.findFirst()

        const settingsData = {
            platformFee: 20, // Force 20%
            minPayout: data.minPayout,
            autoApproveCreators: data.autoApproveCreators ? 1 : 0,
            requireStudioApproval: data.requireStudioApproval ? 1 : 0,
            emailNotifications: data.emailNotifications ? 1 : 0,
            weeklyReports: data.weeklyReports ? 1 : 0,
            maintenanceMode: data.maintenanceMode ? 1 : 0,
            autoMonthlyInvoicing: data.autoMonthlyInvoicing ? 1 : 0,
            discordWebhookUrl: data.discordWebhookUrl || null,
            // Bank EUR
            bankEurBeneficiary: data.bankEurBeneficiary || null,
            bankEurIban: data.bankEurIban || null,
            bankEurBic: data.bankEurBic || null,
            bankEurBankName: data.bankEurBankName || null,
            bankEurBankAddress: data.bankEurBankAddress || null,
            // Bank USD
            bankUsdBeneficiary: data.bankUsdBeneficiary || null,
            bankUsdAccountNumber: data.bankUsdAccountNumber || null,
            bankUsdRoutingNumber: data.bankUsdRoutingNumber || null,
            bankUsdBic: data.bankUsdBic || null,
            bankUsdAccountType: data.bankUsdAccountType || null,
            bankUsdBankName: data.bankUsdBankName || null,
            bankUsdBankAddress: data.bankUsdBankAddress || null,
            updatedAt: new Date()
        }

        if (!existingSettings) {
            await db.insert(globalSettings).values(settingsData)
        } else {
            await db.update(globalSettings)
                .set(settingsData)
                .where(eq(globalSettings.id, existingSettings.id))
        }

        await sendDiscordNotification(
            '⚙️ Settings Updated',
            `Admin ${session.user.email} updated global settings.`,
            'info',
            [
                { name: 'Platform Fee', value: `20%`, inline: true },
                { name: 'Maintenance Mode', value: data.maintenanceMode ? 'ON' : 'OFF', inline: true }
            ]
        )

        revalidatePath('/ops/settings')
        return { success: true }
    } catch (error) {
        return { error: 'Failed to update settings' }
    }
}

