'use server'

import { auth } from '@/auth'
import { db } from '@/lib/db'
import { creatorSocials } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { getInstagramFollowers, getYouTubeSubscribers, getTikTokFollowers } from '@/lib/social-scraper'
import { revalidatePath } from 'next/cache'

export async function refreshCreatorStats(creatorId: string) {
    const session = await auth()
    if (!session || session.user.role !== 'admin') {
        return { error: 'Unauthorized' }
    }

    try {
        const socials = await db.query.creatorSocials.findMany({
            where: eq(creatorSocials.creatorId, creatorId)
        })

        let updatedCount = 0

        for (const social of socials) {
            let followers = null

            if (social.platform === 'instagram') {
                followers = await getInstagramFollowers(social.handle)
            } else if (social.platform === 'youtube') {
                followers = await getYouTubeSubscribers(social.handle)
            } else if (social.platform === 'tiktok') {
                followers = await getTikTokFollowers(social.handle)
            }

            if (followers !== null) {
                await db.update(creatorSocials)
                    .set({
                        followers: followers,
                        updatedAt: new Date()
                    })
                    .where(eq(creatorSocials.id, social.id))
                updatedCount++
            }
        }

        revalidatePath(`/ops/creators/${creatorId}`)
        return { success: true, updatedCount }
    } catch (error) {
        console.error('Error refreshing stats:', error)
        return { error: 'Failed to refresh stats' }
    }
}
