import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { refreshTokens, users } from '@/lib/db/schema'
import { eq, and, gt } from 'drizzle-orm'
import jwt from 'jsonwebtoken'
import crypto from 'crypto'

export async function POST(request: NextRequest) {
  try {
    const { refreshToken } = await request.json()

    if (!refreshToken) {
      return NextResponse.json(
        { error: 'Refresh token requis' },
        { status: 400 }
      )
    }

    // Find valid refresh token
    const tokenRecord = await db.query.refreshTokens.findFirst({
      where: and(
        eq(refreshTokens.token, refreshToken),
        gt(refreshTokens.expiresAt, new Date())
      ),
      with: {
        user: true
      }
    })

    if (!tokenRecord) {
      return NextResponse.json(
        { error: 'Refresh token invalide ou expiré' },
        { status: 401 }
      )
    }

    const user = tokenRecord.user

    // Generate new access token
    const accessToken = jwt.sign(
      {
        userId: user.id,
        email: user.email,
        role: user.role,
      },
      process.env.JWT_SECRET!,
      {
        expiresIn: '15m',
        issuer: 'swiplay-v3'
      }
    )

    // Generate new refresh token
    const newRefreshTokenValue = crypto.randomBytes(64).toString('hex')
    const refreshTokenExpiry = new Date()
    refreshTokenExpiry.setDate(refreshTokenExpiry.getDate() + 30)

    // Delete old refresh token and create new one
    await db.delete(refreshTokens).where(eq(refreshTokens.id, tokenRecord.id))

    await db.insert(refreshTokens).values({
      userId: user.id,
      token: newRefreshTokenValue,
      expiresAt: refreshTokenExpiry,
    })

    return NextResponse.json({
      accessToken,
      refreshToken: newRefreshTokenValue,
    })
  } catch (error) {
    console.error('Refresh token error:', error)
    return NextResponse.json(
      { error: 'Erreur lors du refresh du token' },
      { status: 500 }
    )
  }
}