import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { users, loginCodes } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { rateLimiter } from '@/lib/rate-limit'
import { loginSchema } from '@/lib/validations/auth'
import { getTranslations } from 'next-intl/server'

export async function POST(request: NextRequest) {
  const body = await request.json()
  const locale = body.locale || 'fr'
  const t = await getTranslations({ locale, namespace: 'Api.Errors' })

  try {
    const ip = request.headers.get('x-forwarded-for') ?? '127.0.0.1'

    // Stricter limit for auth: 5 attempts per minute
    if (!rateLimiter.check(5, ip)) {
      return NextResponse.json({ error: t('rate_limit') }, { status: 429 })
    }

    const validation = loginSchema.safeParse(body)

    if (!validation.success) {
      return NextResponse.json({ error: validation.error.errors[0].message }, { status: 400 })
    }

    const { email, role } = validation.data

    // Check if user exists
    const existingUser = await db.query.users.findFirst({
      where: eq(users.email, email)
    })

    // Generate random 6-digit code
    const { randomInt } = await import('crypto')
    const code = randomInt(100000, 999999).toString()
    const expiresAt = new Date(Date.now() + 10 * 60 * 1000) // 10 minutes

    // Save code to database
    await db.insert(loginCodes).values({
      email,
      code,
      role,
      expiresAt,
    })

    // Send email via Nodemailer
    const { sendLoginCode } = await import('@/lib/email')
    try {
      await sendLoginCode(email, code, role, locale)
      // console.log(`Email sent to ${email}`)

      // Discord Log
      const { sendDiscordNotification } = await import('@/lib/discord')
      await sendDiscordNotification(
        '🔑 Code de connexion envoyé',
        `Un code de connexion a été envoyé à **${email}** (${role}).`,
        'info'
      )
    } catch (emailError) {
      console.error('Failed to send email:', emailError)
      return NextResponse.json({ error: t('email_error') }, { status: 500 })
    }

    return NextResponse.json({
      success: true,
      isNewAccount: !existingUser
    })
  } catch (error) {
    console.error('Error sending code:', error)
    return NextResponse.json({ error: t('server_error') }, { status: 500 })
  }
}
