import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { loginCodes, users, refreshTokens } from '@/lib/db/schema'
import { eq, and, gt, isNull } from 'drizzle-orm'
import jwt from 'jsonwebtoken'
import crypto from 'crypto'

export async function POST(request: NextRequest) {
  try {
    const { email, code, role } = await request.json()

    if (!email || !code || !role) {
      return NextResponse.json(
        { error: 'Email, code et rôle requis' },
        { status: 400 }
      )
    }

    // Find valid code
    const validCode = await db.query.loginCodes.findFirst({
      where: and(
        eq(loginCodes.email, email),
        eq(loginCodes.code, code),
        eq(loginCodes.role, role),
        gt(loginCodes.expiresAt, new Date()),
        isNull(loginCodes.usedAt)
      )
    })


    if (!validCode) {
      return NextResponse.json(
        { error: 'Code invalide ou expiré' },
        { status: 401 }
      )
    }

    // Mark code as used
    await db.update(loginCodes)
      .set({ usedAt: new Date() })
      .where(eq(loginCodes.id, validCode.id))

    // Get user - for admin, ignore role in code
    let user
    if (email === 'eyzuro@swiplay.com' || role === 'admin') {
      user = await db.query.users.findFirst({
        where: eq(users.email, email)
      })
    } else {
      user = await db.query.users.findFirst({
        where: and(
          eq(users.email, email),
          eq(users.role, role)
        )
      })
    }

    // Debug: if not found, try to log existing users with same email
    if (!user) {
      const byEmail = await db.query.users.findMany({
        where: eq(users.email, email)
      })
    }
    // If user still not found, create a pending account for non-admin roles
    if (!user) {
      if (role !== 'admin') {
        const insertResult = await db.insert(users).values({
          email,
          role,
          isApproved: 0,
          createdAt: new Date(),
          updatedAt: new Date(),
        })

        // Send notification for new Studio accounts
        if (role === 'studio') {
          const { sendAccessRequestNotification } = await import('@/lib/email')
          // Run in background to not block response
          sendAccessRequestNotification(email, 'Non renseigné').catch(console.error)
        }

        // Try to fetch the newly created user
        user = await db.query.users.findFirst({
          where: and(eq(users.email, email), eq(users.role, role))
        })
      }
    }

    if (!user) {
      return NextResponse.json(
        { error: 'Utilisateur introuvable' },
        { status: 404 }
      )
    }

    // Generate JWT access token (short-lived)
    const accessToken = jwt.sign(
      {
        userId: user.id,
        email: user.email,
        role: user.role,
      },
      process.env.JWT_SECRET!,
      {
        expiresIn: '15m', // 15 minutes
        issuer: 'swiplay-v3'
      }
    )

    // Generate refresh token (long-lived)
    const refreshTokenValue = crypto.randomBytes(64).toString('hex')
    const refreshTokenExpiry = new Date()
    refreshTokenExpiry.setDate(refreshTokenExpiry.getDate() + 30) // 30 days

    // Store refresh token in database
    await db.insert(refreshTokens).values({
      userId: user.id,
      token: refreshTokenValue,
      expiresAt: refreshTokenExpiry,
    })

    // Discord Log
    const { sendDiscordNotification } = await import('@/lib/discord')
    await sendDiscordNotification(
      '✅ Connexion Réussie',
      `L'utilisateur **${user.name || user.email}** (${user.role}) vient de se connecter.`,
      'success',
      [{ name: 'User ID', value: user.id, inline: true }]
    )

    return NextResponse.json({
      message: 'Connexion réussie',
      accessToken,
      refreshToken: refreshTokenValue,
      user: {
        id: user.id,
        email: user.email,
        name: user.name,
        role: user.role,
        isApproved: user.isApproved,
        company: user.company,
      }
    })
  } catch (error) {
    console.error('Verify code error:', error)
    return NextResponse.json(
      { error: 'Erreur lors de la vérification du code' },
      { status: 500 }
    )
  }
}
