import { auth } from '@/auth'
import { db } from '@/lib/db'
import { users } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { NextResponse } from 'next/server'
import { z } from 'zod'

const bankingSchema = z.object({
    iban: z.string().min(15, "IBAN invalide"),
    bic: z.string().min(8, "BIC invalide"),
    bankName: z.string().optional(),
})

export async function POST(req: Request) {
    try {
        const session = await auth()
        if (!session || !session.user) {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
        }

        const body = await req.json()
        const validated = bankingSchema.parse(body)

        await db.update(users)
            .set({
                iban: validated.iban,
                bic: validated.bic,
                // bankName is in creator_profiles if we want to be strict, but schema says users has iban/bic. 
                // Let's check schema again. users has iban/bic.
            })
            .where(eq(users.id, session.user.id))

        return NextResponse.json({ success: true })
    } catch (error) {
        console.error('Banking update error:', error)
        if (error instanceof z.ZodError) {
            return NextResponse.json({ error: error.errors[0].message }, { status: 400 })
        }
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}
