import { auth } from '@/auth'
import { db } from '@/lib/db'
import { users, payoutRequests, globalSettings } from '@/lib/db/schema'
import { eq, desc } from 'drizzle-orm'
import { NextResponse } from 'next/server'
import { sendDiscordNotification } from '@/lib/discord'

export async function POST(req: Request) {
    try {
        const session = await auth()
        if (!session || !session.user) {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
        }

        // 1. Check User Balance & Info
        const user = await db.query.users.findFirst({
            where: eq(users.id, session.user.id),
            columns: {
                balance: true,
                iban: true,
                email: true,
                name: true
            }
        })

        if (!user) return NextResponse.json({ error: 'User not found' }, { status: 404 })
        if (!user.iban) return NextResponse.json({ error: 'Veuillez ajouter votre IBAN avant de demander un retrait' }, { status: 400 })

        const balance = parseFloat(user.balance)

        // 2. Check Min Payout (Default 50€)
        const settings = await db.query.globalSettings.findFirst()
        const minPayout = settings?.minPayout || 50

        if (balance < minPayout) {
            return NextResponse.json({ error: `Le montant minimum de retrait est de ${minPayout}€` }, { status: 400 })
        }

        // 3. Create Payout Request
        await db.transaction(async (tx) => {
            // Create request
            await tx.insert(payoutRequests).values({
                creatorId: session.user.id,
                amount: user.balance,
                status: 'pending'
            })

            // Deduct balance immediately
            await tx.update(users)
                .set({ balance: '0' })
                .where(eq(users.id, session.user.id))
        })

        // 4. Discord notification for admin
        await sendDiscordNotification(
            '💰 Nouvelle Demande de Retrait',
            `${user.name || user.email} demande un retrait de ${balance}€`,
            'warning',
            [
                { name: 'Créateur', value: user.name || 'N/A', inline: true },
                { name: 'Email', value: user.email || 'N/A', inline: true },
                { name: 'Montant', value: `${balance}€`, inline: true },
                { name: 'IBAN', value: user.iban.slice(0, 8) + '****', inline: true }
            ]
        )

        return NextResponse.json({ success: true })
    } catch (error) {
        console.error('Cashout error:', error)
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}

