import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { submissions, campaigns, users } from '@/lib/db/schema'
import { eq, and, count, sum, sql } from 'drizzle-orm'
import { auth } from '@/auth'

export async function GET(request: NextRequest) {
  try {

    const session = await auth()
    if (!session || !session.user || !session.user.id) {
      return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
    }

    if (session.user.role !== 'creator') {
      return NextResponse.json({ error: 'Accès interdit - Créateur requis' }, { status: 403 })
    }

    const creatorId = session.user.id

    // Get total earnings (approved submissions)
    const totalEarningsResult = await db
      .select({ total: sum(submissions.revenue) })
      .from(submissions)
      .where(and(eq(submissions.creatorId, creatorId), eq(submissions.status, 'approved')))

    const totalEarnings = totalEarningsResult[0]?.total ? parseFloat(totalEarningsResult[0].total) : 0

    // Get this month's earnings
    const thisMonthStart = new Date()
    thisMonthStart.setDate(1)
    thisMonthStart.setHours(0, 0, 0, 0)

    const thisMonthEarningsResult = await db
      .select({ total: sum(submissions.revenue) })
      .from(submissions)
      .where(and(
        eq(submissions.creatorId, creatorId),
        eq(submissions.status, 'approved'),
        sql`${submissions.submittedAt} >= ${thisMonthStart}`
      ))

    const thisMonthEarnings = thisMonthEarningsResult[0]?.total ? parseFloat(thisMonthEarningsResult[0].total) : 0

    // Get total views
    const totalViewsResult = await db
      .select({ total: sum(submissions.viewsCount) })
      .from(submissions)
      .where(eq(submissions.creatorId, creatorId))

    const totalViews = totalViewsResult[0]?.total || 0

    // Get this month's views
    const thisMonthViewsResult = await db
      .select({ total: sum(submissions.viewsCount) })
      .from(submissions)
      .where(and(
        eq(submissions.creatorId, creatorId),
        sql`${submissions.submittedAt} >= ${thisMonthStart}`
      ))

    const thisMonthViews = thisMonthViewsResult[0]?.total || 0

    // Get active campaigns count (campaigns with creator's submissions)
    const activeCampaignsResult = await db
      .select({ count: count() })
      .from(submissions)
      .where(and(
        eq(submissions.creatorId, creatorId),
        sql`${campaigns.endDate} > now() OR ${campaigns.endDate} IS NULL`
      ))
      .leftJoin(campaigns, eq(submissions.campaignId, campaigns.id))

    const activeCampaigns = activeCampaignsResult[0]?.count || 0

    // Get submission counts
    const pendingSubmissionsResult = await db
      .select({ count: count() })
      .from(submissions)
      .where(and(eq(submissions.creatorId, creatorId), eq(submissions.status, 'pending')))

    const approvedSubmissionsResult = await db
      .select({ count: count() })
      .from(submissions)
      .where(and(eq(submissions.creatorId, creatorId), eq(submissions.status, 'approved')))

    const rejectedSubmissionsResult = await db
      .select({ count: count() })
      .from(submissions)
      .where(and(eq(submissions.creatorId, creatorId), eq(submissions.status, 'rejected')))

    const pendingSubmissions = pendingSubmissionsResult[0]?.count || 0
    const approvedSubmissions = approvedSubmissionsResult[0]?.count || 0
    const rejectedSubmissions = rejectedSubmissionsResult[0]?.count || 0

    // Get recent submissions
    const recentSubmissions = await db
      .select({
        id: submissions.id,
        campaignTitle: campaigns.title,
        platform: submissions.platform,
        viewsCount: submissions.viewsCount,
        revenue: submissions.revenue,
        status: submissions.status,
        submittedAt: submissions.submittedAt,
      })
      .from(submissions)
      .leftJoin(campaigns, eq(submissions.campaignId, campaigns.id))
      .where(eq(submissions.creatorId, creatorId))
      .orderBy(sql`${submissions.submittedAt} desc`)
      .limit(5)

    // Get available campaigns (active campaigns not yet submitted to by this creator)
    const availableCampaigns = await db
      .select({
        id: campaigns.id,
        title: campaigns.title,
        studioName: users.name,
        pricePerView: campaigns.pricePerView,
        budget: campaigns.budget,
        endDate: campaigns.endDate,
      })
      .from(campaigns)
      .leftJoin(users, eq(campaigns.studioId, users.id))
      .where(and(
        eq(campaigns.status, 'active'),
        sql`${campaigns.endDate} > now() OR ${campaigns.endDate} IS NULL`
      ))
      .orderBy(sql`${campaigns.createdAt} desc`)
      .limit(5)

    return NextResponse.json({
      stats: {
        totalEarnings,
        thisMonthEarnings,
        totalViews,
        thisMonthViews,
        activeCampaigns,
        pendingSubmissions,
        approvedSubmissions,
        rejectedSubmissions,
      },
      recentSubmissions: recentSubmissions.map(s => ({
        id: s.id,
        campaign: s.campaignTitle || 'Unknown Campaign',
        platform: s.platform,
        views: s.viewsCount,
        revenue: parseFloat(s.revenue),
        status: s.status,
        date: s.submittedAt?.toISOString().split('T')[0] || '',
      })),
      availableCampaigns: availableCampaigns.map(c => ({
        id: c.id,
        title: c.title,
        studio: c.studioName || 'Unknown Studio',
        cpm: parseFloat(c.pricePerView) * 1000, // Convert to CPM
        budget: parseFloat(c.budget),
        deadline: c.endDate?.toISOString().split('T')[0] || null,
      })),
    })
  } catch (error) {
    console.error('Error fetching dashboard data:', error)
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 })
  }
}