import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { submissions, payments, campaigns, users } from '@/lib/db/schema'
import { eq, and, sum, sql } from 'drizzle-orm'
import { auth } from '@/auth'

export async function GET(request: NextRequest) {
  try {
    const session = await auth()
    if (!session || !session.user || !session.user.id) {
      return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
    }

    if (session.user.role !== 'creator') {
      return NextResponse.json({ error: 'Accès interdit - Créateur requis' }, { status: 403 })
    }

    const creatorId = session.user.id

    // Get total earnings (sum of revenue from approved submissions)
    const totalEarningsResult = await db
      .select({ total: sum(submissions.revenue) })
      .from(submissions)
      .where(and(eq(submissions.creatorId, creatorId), eq(submissions.status, 'approved')))

    const totalEarnings = totalEarningsResult[0]?.total ? parseFloat(totalEarningsResult[0].total) : 0

    // Get pending earnings (sum of revenue from pending submissions)
    const pendingEarningsResult = await db
      .select({ total: sum(submissions.revenue) })
      .from(submissions)
      .where(and(eq(submissions.creatorId, creatorId), eq(submissions.status, 'pending')))

    const pendingEarnings = pendingEarningsResult[0]?.total ? parseFloat(pendingEarningsResult[0].total) : 0

    // Get paid earnings (sum of paid payments)
    const paidEarningsResult = await db
      .select({ total: sum(payments.amount) })
      .from(payments)
      .where(and(eq(payments.creatorId, creatorId), eq(payments.status, 'paid')))

    const paidEarnings = paidEarningsResult[0]?.total ? parseFloat(paidEarningsResult[0].total) : 0

    // Get transactions (recent submissions with campaign info)
    const transactions = await db
      .select({
        id: submissions.id,
        campaignTitle: campaigns.title,
        revenue: submissions.revenue,
        status: submissions.status,
        submittedAt: submissions.submittedAt,
        viewsCount: submissions.viewsCount,
        platform: submissions.platform,
      })
      .from(submissions)
      .leftJoin(campaigns, eq(submissions.campaignId, campaigns.id))
      .where(eq(submissions.creatorId, creatorId))
      .orderBy(sql`${submissions.submittedAt} desc`)
      .limit(50)

    // Get monthly earnings data (last 12 months)
    const monthlyData = await db
      .select({
        month: sql<string>`to_char(${submissions.submittedAt}, 'YYYY-MM')`,
        earnings: sum(submissions.revenue),
      })
      .from(submissions)
      .where(and(
        eq(submissions.creatorId, creatorId),
        eq(submissions.status, 'approved'),
        sql`${submissions.submittedAt} >= now() - interval '12 months'`
      ))
      .groupBy(sql`to_char(${submissions.submittedAt}, 'YYYY-MM')`)
      .orderBy(sql`to_char(${submissions.submittedAt}, 'YYYY-MM')`)

    const formattedMonthlyData = monthlyData.map(item => ({
      month: item.month,
      earnings: parseFloat(item.earnings || '0'),
    }))

    // Get user banking info
    const user = await db.query.users.findFirst({
      where: eq(users.id, creatorId),
      columns: {
        iban: true,
        bic: true,
        balance: true,
      }
    })

    return NextResponse.json({
      userData: {
        iban: user?.iban || '',
        bic: user?.bic || '',
        balance: user?.balance || '0',
      },
      totalEarnings,
      pendingEarnings,
      paidEarnings,
      transactions: transactions.map(t => ({
        id: t.id,
        campaign: t.campaignTitle || 'Unknown Campaign',
        amount: parseFloat(t.revenue),
        status: t.status,
        date: t.submittedAt?.toISOString().split('T')[0] || '',
        views: t.viewsCount,
        platform: t.platform,
      })),
      monthlyData: formattedMonthlyData,
    })
  } catch (error) {
    console.error('Error fetching earnings:', error)
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 })
  }
}