import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { users, creatorProfiles, creatorSocials } from '@/lib/db/schema'
import { eq, and } from 'drizzle-orm'
import { auth } from '@/auth'

// GET - Fetch creator profile
export async function GET(request: NextRequest) {
  try {
    const session = await auth()

    if (!session || !session.user) {
      return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
    }

    if (session.user.role !== 'creator') {
      return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
    }

    const userId = session.user.id

    const user = await db.query.users.findFirst({
      where: eq(users.id, userId)
    })

    if (!user) {
      return NextResponse.json({ error: 'Utilisateur introuvable' }, { status: 404 })
    }

    // Get creator profile data
    const profile = await db.query.creatorProfiles.findFirst({
      where: eq(creatorProfiles.creatorId, userId)
    })

    // Check for verified socials
    const verifiedSocialsCount = await db.query.creatorSocials.findMany({
      where: and(
        eq(creatorSocials.creatorId, userId),
        eq(creatorSocials.verificationStatus, 'approved')
      )
    })

    return NextResponse.json({
      id: user.id,
      email: user.email,
      name: user.name,
      bio: profile?.bio || '',
      avatar: user.image || '',
      role: user.role,
      isApproved: user.isApproved,
      hasVerifiedSocials: verifiedSocialsCount.length > 0,
      socials: {
        instagram: profile?.instagram || '',
        youtube: profile?.youtube || '',
        tiktok: profile?.tiktok || ''
      },
      stats: {
        followers: profile?.followers || 0,
        avgViews: profile?.avgViews || 0,
        engagement: profile?.followers && profile?.avgViews
          ? Math.min(((profile.avgViews / profile.followers) * 100), 100).toFixed(1)
          : 0
      }
    })
  } catch (error) {
    console.error('Get profile error:', error)
    return NextResponse.json(
      { error: 'Erreur lors de la récupération du profil' },
      { status: 500 }
    )
  }
}

// PUT - Update creator profile
export async function PUT(request: NextRequest) {
  try {
    const session = await auth()

    if (!session || !session.user) {
      return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
    }

    if (session.user.role !== 'creator') {
      return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
    }

    const userId = session.user.id

    const data = await request.json()
    const { name, bio, avatar, socials } = data

    // Update user in database (only name for now, bio/avatar will be added later)
    await db.update(users)
      .set({
        name: name || null,
        image: avatar || null,
        updatedAt: new Date()
      })
      .where(eq(users.id, userId))

    // Update or create creator profile
    const existingProfile = await db.query.creatorProfiles.findFirst({
      where: eq(creatorProfiles.creatorId, userId)
    })

    if (existingProfile) {
      await db.update(creatorProfiles)
        .set({
          instagram: socials?.instagram || null,
          youtube: socials?.youtube || null,
          tiktok: socials?.tiktok || null,
          bio: bio || null,
          updatedAt: new Date()
        })
        .where(eq(creatorProfiles.creatorId, userId))
    } else {
      await db.insert(creatorProfiles).values({
        creatorId: userId,
        instagram: socials?.instagram || null,
        youtube: socials?.youtube || null,
        tiktok: socials?.tiktok || null,
        bio: bio || null,
      })
    }

    return NextResponse.json({
      message: 'Profil mis à jour avec succès'
    })
  } catch (error) {
    console.error('Update profile error:', error)
    return NextResponse.json(
      { error: 'Erreur lors de la mise à jour du profil' },
      { status: 500 }
    )
  }
}
