import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { db } from '@/lib/db'
import { creatorSocials, accounts } from '@/lib/db/schema'
import { eq, and } from 'drizzle-orm'

export async function POST(req: NextRequest) {
    const session = await auth()
    if (!session?.user?.id) return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })

    const { provider } = await req.json()

    try {
        // 1. Remove from creatorSocials
        await db.delete(creatorSocials)
            .where(and(
                eq(creatorSocials.creatorId, session.user.id),
                eq(creatorSocials.platform, provider)
            ))

        // 2. Remove from accounts (NextAuth) to force re-auth next time
        // Map platform names if they differ (e.g. 'youtube' -> 'google')
        let authProvider = provider
        if (provider === 'youtube') authProvider = 'google'

        await db.delete(accounts)
            .where(and(
                eq(accounts.userId, session.user.id),
                eq(accounts.provider, authProvider)
            ))

        // Discord Log
        const { sendDiscordNotification } = await import('@/lib/discord')
        await sendDiscordNotification(
            '💔 Réseau Social Déconnecté',
            `Le créateur **${session.user.name || session.user.email}** a déconnecté **${provider}**.`,
            'warning',
            [{ name: 'User ID', value: session.user.id, inline: true }]
        )

        return NextResponse.json({ success: true })
    } catch (error) {
        console.error('Disconnect Error:', error)
        return NextResponse.json({ error: 'Failed to disconnect' }, { status: 500 })
    }
}
