import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { getInstagramFollowers, getTikTokFollowers, getYouTubeSubscribers } from '@/lib/social-scraper'

export async function POST(req: NextRequest) {
    try {
        const session = await auth()
        if (!session || !session.user) {
            return new NextResponse('Unauthorized', { status: 401 })
        }

        const body = await req.json()
        const { platform, username } = body

        if (!platform || !username) {
            return NextResponse.json({ error: 'Platform and username are required' }, { status: 400 })
        }

        let result = null

        switch (platform) {
            case 'instagram':
                result = await getInstagramFollowers(username)
                break
            case 'tiktok':
                result = await getTikTokFollowers(username)
                break
            case 'youtube':
                result = await getYouTubeSubscribers(username)
                break
            default:
                return NextResponse.json({ error: 'Invalid platform' }, { status: 400 })
        }

        if (!result) {
            return NextResponse.json({ error: 'Could not verify account. Please check the username.' }, { status: 404 })
        }

        return NextResponse.json({
            success: true,
            data: {
                platform,
                handle: username,
                followers: result.followers,
                avatarUrl: result.avatarUrl
            }
        })

    } catch (error) {
        console.error('Verification error:', error)
        return new NextResponse('Internal Server Error', { status: 500 })
    }
}
