import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { users, creatorProfiles } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import nodemailer from 'nodemailer'
import jwt from 'jsonwebtoken'

interface JWTPayload {
  userId: string
  email: string
  role: string
}

export async function POST(request: NextRequest) {
  try {
    const authHeader = request.headers.get('authorization')
    if (!authHeader) {
      return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
    }

    // Decode token to get user ID
    const token = authHeader.replace('Bearer ', '')
    const decoded = jwt.verify(token, process.env.JWT_SECRET!) as JWTPayload
    const userId = decoded.userId

    const body = await request.json()
    const { tiktok, youtube, instagram } = body

    // Get user email for notification
    const user = await db.query.users.findFirst({
      where: eq(users.id, userId)
    })

    if (!user) {
      return NextResponse.json({ error: 'Utilisateur introuvable' }, { status: 404 })
    }

    // Check if profile already exists
    const existingProfile = await db.query.creatorProfiles.findFirst({
      where: eq(creatorProfiles.creatorId, userId)
    })

    if (existingProfile) {
      // Update existing profile
      await db
        .update(creatorProfiles)
        .set({
          tiktok: tiktok || null,
          youtube: youtube || null,
          instagram: instagram || null,
          updatedAt: new Date(),
        })
        .where(eq(creatorProfiles.creatorId, userId))
    } else {
      // Create new profile
      await db.insert(creatorProfiles).values({
        creatorId: userId,
        tiktok: tiktok || null,
        youtube: youtube || null,
        instagram: instagram || null,
      })
    }

    // Set user to pending approval
    await db
      .update(users)
      .set({ isApproved: 0 })
      .where(eq(users.id, userId))

    // Send notification email to admin
    const transporter = nodemailer.createTransport({
      host: process.env.SMTP_HOST,
      port: parseInt(process.env.SMTP_PORT || '587'),
      secure: false,
      auth: {
        user: process.env.SMTP_USER,
        pass: process.env.SMTP_PASS,
      },
    })

    const socialsList = [
      tiktok && `TikTok: @${tiktok}`,
      youtube && `YouTube: @${youtube}`,
      instagram && `Instagram: @${instagram}`,
    ].filter(Boolean).join('\n')

    await transporter.sendMail({
      from: process.env.SMTP_FROM,
      to: process.env.ADMIN_EMAIL || 'contact@swiplay.com',
      subject: `Nouvelle candidature créateur - ${user.email}`,
      html: `
        <!DOCTYPE html>
        <html>
        <head>
          <style>
            body {
              font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
              line-height: 1.6;
              color: #333;
            }
            .container {
              max-width: 600px;
              margin: 0 auto;
              padding: 20px;
            }
            .header {
              background: linear-gradient(135deg, #8a2be2 0%, #4169e1 100%);
              color: white;
              padding: 30px;
              border-radius: 12px 12px 0 0;
              text-align: center;
            }
            .content {
              background: #f9fafb;
              padding: 30px;
              border-radius: 0 0 12px 12px;
            }
            .socials {
              background: white;
              padding: 20px;
              border-radius: 8px;
              margin: 20px 0;
            }
            .button {
              display: inline-block;
              background: #8a2be2;
              color: white;
              padding: 12px 30px;
              text-decoration: none;
              border-radius: 8px;
              margin: 10px 5px;
            }
            .footer {
              text-align: center;
              margin-top: 30px;
              color: #666;
              font-size: 14px;
            }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Nouvelle candidature créateur</h1>
            </div>
            <div class="content">
              <h3>Informations du créateur :</h3>
              <p><strong>Email :</strong> ${user.email}</p>
              
              <div class="socials">
                <h4>Réseaux sociaux :</h4>
                <pre>${socialsList || 'Aucun réseau renseigné'}</pre>
              </div>
              
              <p>Veuillez vérifier le profil du créateur et approuver ou refuser sa candidature depuis le panneau admin.</p>
              
              <div style="text-align: center; margin-top: 30px;">
                <a href="https://beta.swiplay.com/ops/creators" class="button">Voir dans l'admin</a>
              </div>
            </div>
            <div class="footer">
              <p>Swiplay - Notification automatique</p>
            </div>
          </div>
        </body>
        </html>
      `,
    })

    return NextResponse.json({
      success: true,
      message: 'Candidature envoyée avec succès'
    })
  } catch (error) {
    console.error('Error submitting creator socials:', error)
    return NextResponse.json(
      { error: 'Erreur lors de l\'envoi' },
      { status: 500 }
    )
  }
}
