import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { handleApiError, createApiResponse } from '@/lib/api-response'
import { db } from '@/lib/db'
import { users } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import fs from 'fs/promises'
import path from 'path'

const UPLOAD_DIR = path.join(process.cwd(), 'public', 'uploads', 'avatars')
const MAX_FILE_SIZE = 5 * 1024 * 1024 // 5MB
const ALLOWED_TYPES = ['image/jpeg', 'image/png', 'image/gif', 'image/webp']

export async function POST(request: NextRequest) {
  try {
    const session = await auth()
    if (!session || !session.user) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    if (session.user.role !== 'creator') {
      return NextResponse.json({ error: 'Forbidden' }, { status: 403 })
    }

    const userId = session.user.id

    let formData;
    try {
      formData = await request.formData()
    } catch (e) {
      console.error('Error parsing FormData:', e)
      return NextResponse.json(
        { error: 'Invalid form data' },
        { status: 400 }
      )
    }

    const file = formData.get('avatar') as File

    // Debug logging removed for production

    if (!file) {
      return NextResponse.json(
        { error: 'Fichier avatar requis' },
        { status: 400 }
      )
    }

    // Vérifier la taille du fichier
    if (file.size > MAX_FILE_SIZE) {
      // console.log('File too large:', file.size)
      return NextResponse.json(
        { error: 'Fichier trop volumineux (max 5MB)' },
        { status: 400 }
      )
    }

    // Vérifier le type de fichier
    if (!ALLOWED_TYPES.includes(file.type)) {
      // console.log('Invalid file type:', file.type)
      return NextResponse.json(
        { error: 'Type de fichier non autorisé (JPEG, PNG, GIF, WebP uniquement)' },
        { status: 400 }
      )
    }

    // Créer le répertoire d'upload s'il n'existe pas
    await fs.mkdir(UPLOAD_DIR, { recursive: true })

    // Générer un nom de fichier unique
    const fileExtension = file.name.split('.').pop()?.toLowerCase() || 'jpg'
    const uniqueFileName = `${userId}_${Date.now()}.${fileExtension}`
    const filePath = path.join(UPLOAD_DIR, uniqueFileName)

    // Convertir le fichier en buffer et l'écrire
    const arrayBuffer = await file.arrayBuffer()
    const buffer = Buffer.from(arrayBuffer)
    await fs.writeFile(filePath, buffer)

    // Construire l'URL publique
    const publicUrl = `/api/uploads/avatars/${uniqueFileName}`

    // Mettre à jour l'avatar dans la base de données
    await db.update(users)
      .set({
        image: publicUrl,
        updatedAt: new Date()
      })
      .where(eq(users.id, userId))

    return createApiResponse({
      avatarUrl: publicUrl,
      fileName: uniqueFileName,
      size: file.size,
      type: file.type,
      uploadedAt: new Date().toISOString()
    })

  } catch (error) {
    return handleApiError(error)
  }
}