import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { submissions, invoices, users, campaigns } from '@/lib/db/schema'
import { eq, and, isNull, inArray, gt } from 'drizzle-orm'
import { generateMandateInvoice } from '@/lib/pdf/generator'
import { sendDiscordNotification } from '@/lib/discord'

export async function GET(req: NextRequest) {
    // 1. Security Check
    const authHeader = req.headers.get('authorization')
    if (authHeader !== `Bearer ${process.env.CRON_SECRET}`) {
        return new NextResponse('Unauthorized', { status: 401 })
    }

    try {
        // 1.5 Check Global Settings - REMOVED (Forced by system)
        // const settings = await db.query.globalSettings.findFirst()
        // if (!settings || settings.autoMonthlyInvoicing !== 1) { ... }

        // 2. Fetch pending submissions (approved but not invoiced)
        // AND have views > 0
        const pendingSubmissions = await db.query.submissions.findMany({
            where: and(
                eq(submissions.status, 'approved'),
                isNull(submissions.invoiceId),
                gt(submissions.viewsCount, 0)
            ),
            with: {
                creator: true,
                campaign: {
                    with: {
                        studio: true
                    }
                }
            },
            limit: 100 // Process in batches to avoid OOM
        })

        if (pendingSubmissions.length === 0) {
            return NextResponse.json({ message: 'No pending submissions to invoice.' })
        }

        // 3. Group by Creator -> Studio
        const invoiceGroups = new Map<string, typeof pendingSubmissions>()

        for (const sub of pendingSubmissions) {
            const key = `${sub.creatorId}-${sub.campaign.studioId}`
            if (!invoiceGroups.has(key)) {
                invoiceGroups.set(key, [])
            }
            invoiceGroups.get(key)!.push(sub)
        }

        let invoicesGenerated = 0

        // 4. Process each group
        for (const [key, subs] of invoiceGroups) {
            const creator = subs[0].creator
            const studio = subs[0].campaign.studio
            const totalAmount = subs.reduce((sum, s) => sum + parseFloat(s.revenue), 0)

            // Generate Invoice Number (YYYYMM-CREATORID-STUDIOID)
            const date = new Date()
            const invoiceNumber = `INV-${date.getFullYear()}${(date.getMonth() + 1).toString().padStart(2, '0')}-${creator.id.slice(0, 4)}-${studio.id.slice(0, 4)}`

            // Generate PDF
            const pdfPath = await generateMandateInvoice({
                invoiceNumber,
                date,
                studio: {
                    name: studio.name || studio.email,
                    address: studio.company || 'Adresse non renseignée', // Should fetch from profile
                    email: studio.email
                },
                creator: {
                    name: creator.name || creator.email,
                    address: 'Adresse non renseignée', // Should fetch from profile
                    email: creator.email
                },
                campaignTitle: `Campagnes du mois (${subs.length} soumissions)`,
                amount: totalAmount * 100 // PDF generator expects cents? No, let's check.
            })

            // Check PDF generator amount expectation
            // In generateMandateInvoice: text(`${(data.amount / 100).toFixed(2)} €`
            // So it expects cents.
            // revenue in DB is decimal string (e.g. "10.50").
            // totalAmount is float.
            // So we need to pass totalAmount * 100.

            // Create Invoice Record
            const [newInvoice] = await db.insert(invoices).values({
                creatorId: creator.id,
                studioId: studio.id,
                amount: totalAmount.toString(),
                pdfUrl: pdfPath, // In real app, upload to S3 and store URL
                invoiceNumber,
                periodStart: new Date(date.getFullYear(), date.getMonth() - 1, 1), // Previous month
                periodEnd: new Date(date.getFullYear(), date.getMonth(), 0),
            }).returning()

            // Update Submissions
            await db.update(submissions)
                .set({ invoiceId: newInvoice.id })
                .where(inArray(submissions.id, subs.map(s => s.id)))

            // Notify Discord
            await sendDiscordNotification(
                '📄 Facture Mensuelle Générée',
                `Facture **${invoiceNumber}** générée pour **${creator.name}** vers **${studio.name}**.`,
                'info',
                [
                    { name: 'Montant', value: `${totalAmount.toFixed(2)}€`, inline: true },
                    { name: 'Soumissions', value: subs.length.toString(), inline: true }
                ]
            )

            invoicesGenerated++
        }

        return NextResponse.json({
            success: true,
            invoicesGenerated,
            submissionsProcessed: pendingSubmissions.length
        })

    } catch (error) {
        console.error('[CRON_INVOICING_ERROR]', error)
        await sendDiscordNotification(
            '❌ Erreur Cron Facturation',
            `Erreur lors de la génération des factures mensuelles : ${error instanceof Error ? error.message : 'Inconnue'}`,
            'error'
        )
        return new NextResponse('Internal Error', { status: 500 })
    }
}
