import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { invoices } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { sendDiscordNotification } from '@/lib/discord'

export async function GET(req: NextRequest) {
    // 1. Security Check
    const authHeader = req.headers.get('authorization')
    if (process.env.CRON_SECRET && authHeader !== `Bearer ${process.env.CRON_SECRET}`) {
        return new NextResponse('Unauthorized', { status: 401 })
    }

    try {
        // 2. Fetch pending invoices just to report
        const pendingInvoices = await db.query.invoices.findMany({
            where: eq(invoices.status, 'pending'),
            with: {
                creator: true
            },
            limit: 50
        })

        if (pendingInvoices.length === 0) {
            return NextResponse.json({ message: 'No pending invoices.' })
        }

        // AUTOMATIC PAYOUTS DISABLED - MANUAL APPROVAL REQUIRED
        // We just log or notify if needed, but for now we do nothing.
        // The Admin Dashboard will handle the actual transfer.

        // console.log(`[CRON_PAYOUTS] Found ${pendingInvoices.length} pending invoices. Waiting for manual approval.`)

        return NextResponse.json({
            success: true,
            message: `Found ${pendingInvoices.length} pending invoices. Auto-payout is disabled.`,
            pendingCount: pendingInvoices.length
        })

    } catch (error) {
        console.error('[CRON_PAYOUTS_ERROR]', error)
        return new NextResponse('Internal Error', { status: 500 })
    }
}
