import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { notificationQueue } from '@/lib/db/schema'
import { eq, lt, and, or } from 'drizzle-orm'

export async function GET(req: NextRequest) {
  // Simple security check (Cron secret)
  const authHeader = req.headers.get('authorization')
  if (process.env.CRON_SECRET && authHeader !== `Bearer ${process.env.CRON_SECRET}`) {
    return new NextResponse('Unauthorized', { status: 401 })
  }

  try {
    // Fetch pending notifications (limit 10 to avoid timeout)
    const pending = await db.query.notificationQueue.findMany({
      where: or(
        eq(notificationQueue.status, 'pending'),
        and(eq(notificationQueue.status, 'failed'), lt(notificationQueue.attempts, 3))
      ),
      limit: 10,
    })

    if (pending.length === 0) {
      return NextResponse.json({ processed: 0 })
    }

    let processed = 0

    for (const item of pending) {
      try {
        // Mark as processing
        await db.update(notificationQueue)
          .set({ status: 'processing', attempts: item.attempts + 1 })
          .where(eq(notificationQueue.id, item.id))

        if (item.type === 'discord') {
          await processDiscordNotification(JSON.parse(item.payload))
        }

        // Mark as completed
        await db.update(notificationQueue)
          .set({ status: 'completed', updatedAt: new Date() })
          .where(eq(notificationQueue.id, item.id))
        
        processed++
      } catch (error) {
        console.error(`Failed to process notification ${item.id}:`, error)
        // Mark as failed
        await db.update(notificationQueue)
          .set({ 
            status: 'failed', 
            lastError: error instanceof Error ? error.message : 'Unknown error',
            nextRetry: new Date(Date.now() + 5 * 60 * 1000) // Retry in 5 mins
          })
          .where(eq(notificationQueue.id, item.id))
      }
    }

    return NextResponse.json({ processed })
  } catch (error) {
    console.error('Error in notification cron:', error)
    return new NextResponse('Internal Error', { status: 500 })
  }
}

async function processDiscordNotification(data: any) {
    let webhookUrl = process.env.DISCORD_WEBHOOK_URL

    // We could fetch global settings here if needed, but for performance let's stick to env or cache
    // If we really need dynamic webhook URL from DB, we can do it.
    // Let's do a quick check for global settings if env is missing or if we want to support override
    if (!webhookUrl) {
       const settings = await db.query.globalSettings.findFirst()
       if (settings?.discordWebhookUrl) webhookUrl = settings.discordWebhookUrl
    }

    if (!webhookUrl) throw new Error('No Discord Webhook URL configured')

    const colors = {
        info: 3447003,
        success: 5763719,
        warning: 16776960,
        error: 15548997,
    }

    const payload = {
        embeds: [
            {
                title: data.title,
                description: data.message,
                color: colors[data.type as keyof typeof colors] || colors.info,
                fields: data.fields || [],
                timestamp: new Date().toISOString(),
                footer: {
                    text: 'Swiplay System Notification',
                },
            },
        ],
    }

    const res = await fetch(webhookUrl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload),
    })

    if (!res.ok) {
        throw new Error(`Discord API responded with ${res.status}: ${res.statusText}`)
    }
}
