import { NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { submissions } from '@/lib/db/schema'
import { eq, and } from 'drizzle-orm'
import { sendDiscordNotification } from '@/lib/discord'

export async function GET(request: Request) {
    try {
        // Find approved submissions
        const activeSubmissions = await db.query.submissions.findMany({
            where: eq(submissions.status, 'approved'),
            with: {
                campaign: true,
                creator: true
            },
            limit: 50 // Process in batches to avoid OOM
        })

        if (activeSubmissions.length === 0) {
            return NextResponse.json({ message: 'No active submissions to update' })
        }

        const results = []

        for (const submission of activeSubmissions) {
            // SIMULATION: In a real app, we would call YouTube/TikTok API here
            // using submission.videoUrl and creator's tokens.

            // For demo/MVP, we will simulate view growth if the video is "posted" (approved)
            // We'll add random views (0-1000) every run
            const newViews = Math.floor(Math.random() * 500)
            const currentViews = submission.viewsCount || 0
            const updatedViews = currentViews + newViews

            // Calculate revenue (Views * PricePerView)
            const pricePerView = Number(submission.campaign.pricePerView)
            const revenue = (updatedViews * pricePerView).toFixed(2)

            await db.update(submissions)
                .set({
                    viewsCount: updatedViews,
                    revenue: revenue,
                    updatedAt: new Date()
                })
                .where(eq(submissions.id, submission.id))

            results.push({
                id: submission.id,
                views: updatedViews,
                revenue: revenue
            })
        }

        // Log summary to Discord if significant updates
        if (results.length > 0) {
            // Only log occasionally or summary to avoid spam
            // console.log(`Updated metrics for ${results.length} submissions`)
        }

        return NextResponse.json({
            success: true,
            count: results.length,
            updates: results
        })

    } catch (error) {
        console.error('Metrics update cron error:', error)
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}
