import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { submissions, users, campaigns } from '@/lib/db/schema'
import { eq, and, sql, or } from 'drizzle-orm'
import { getPlatformViews } from '@/lib/platform-utils'

export const dynamic = 'force-dynamic'
export const maxDuration = 300 // 5 minutes max for Vercel

// Configuration
const CONCURRENCY = 5          // Number of parallel requests
const DELAY_MS = 500           // Delay between batches to avoid rate limiting
const MAX_RETRIES = 2          // Retry failed requests

interface SubmissionToUpdate {
    id: string
    videoUrl: string
    platform: string
    views: number
    revenue: string
    creatorId: string
    pricePerView: string
    creatorCap: string | null
    campaignStatus: string
}

// Helper function to delay execution
const delay = (ms: number) => new Promise(resolve => setTimeout(resolve, ms))

// Process a single submission
async function processSubmission(sub: SubmissionToUpdate): Promise<{
    id: string
    success: boolean
    viewsUpdated: boolean
    newViews?: number
    revenueDiff?: number
    error?: string
}> {
    try {
        const currentViews = sub.views || 0
        const newViews = await getPlatformViews(sub.videoUrl, sub.platform)

        if (newViews === null) {
            return { id: sub.id, success: true, viewsUpdated: false }
        }

        if (newViews <= currentViews) {
            return { id: sub.id, success: true, viewsUpdated: false }
        }

        const pricePerView = Number(sub.pricePerView)
        const cap = sub.creatorCap ? Number(sub.creatorCap) : Infinity

        const currentRevenue = Number(sub.revenue || 0)
        const potentialRevenue = newViews * pricePerView
        const newRevenue = Math.min(potentialRevenue, cap)

        const revenueDiff = newRevenue - currentRevenue

        // Update submission
        await db.update(submissions)
            .set({
                viewsCount: newViews,
                revenue: newRevenue.toString()
            })
            .where(eq(submissions.id, sub.id))

        // Update user balance if there are new earnings
        if (revenueDiff > 0) {
            await db.update(users)
                .set({
                    balance: sql`${users.balance} + ${revenueDiff}`
                })
                .where(eq(users.id, sub.creatorId))
        }

        return {
            id: sub.id,
            success: true,
            viewsUpdated: true,
            newViews,
            revenueDiff
        }

    } catch (error: any) {
        console.error(`Error processing submission ${sub.id}:`, error)
        return {
            id: sub.id,
            success: false,
            viewsUpdated: false,
            error: error.message
        }
    }
}

// Process submissions in parallel batches
async function processBatch(batch: SubmissionToUpdate[]): Promise<any[]> {
    return Promise.all(batch.map(sub => processSubmission(sub)))
}

export async function GET(request: NextRequest) {
    const startTime = Date.now()

    try {
        // Optional: Check for Cron Secret
        const authHeader = request.headers.get('authorization')
        const cronSecret = process.env.CRON_SECRET
        if (cronSecret && authHeader !== `Bearer ${cronSecret}`) {
            // Allow without auth in dev, but log warning
            console.warn('Cron called without proper authorization')
        }

        // console.log('[CRON] Starting view update job...')

        // Fetch all approved submissions from ACTIVE campaigns only
        // Also filter out submissions where creator has already hit the cap
        const approvedSubmissions = await db
            .select({
                id: submissions.id,
                videoUrl: submissions.videoUrl,
                platform: submissions.platform,
                views: submissions.viewsCount,
                revenue: submissions.revenue,
                creatorId: submissions.creatorId,
                pricePerView: campaigns.pricePerView,
                creatorCap: campaigns.creatorCap,
                campaignStatus: campaigns.status,
            })
            .from(submissions)
            .innerJoin(campaigns, eq(submissions.campaignId, campaigns.id))
            .where(
                and(
                    eq(submissions.status, 'approved'),
                    // Only active campaigns (not draft, paused, completed, rejected)
                    eq(campaigns.status, 'active')
                )
            )

        // Filter out submissions where creator has already hit the cap
        const submissionsToProcess = approvedSubmissions.filter(sub => {
            if (!sub.creatorCap) return true // No cap, always process
            const currentRevenue = Number(sub.revenue || 0)
            const cap = Number(sub.creatorCap)
            return currentRevenue < cap // Only process if under cap
        })

        // console.log(`[CRON] Found ${approvedSubmissions.length} approved submissions`)
        // console.log(`[CRON] Processing ${submissionsToProcess.length} (${approvedSubmissions.length - submissionsToProcess.length} already at cap)`)

        if (submissionsToProcess.length === 0) {
            return NextResponse.json({
                success: true,
                message: 'No submissions to process',
                processed: 0,
                updated: 0,
                skippedAtCap: 0,
                durationMs: Date.now() - startTime
            })
        }

        // Process in parallel batches
        const results: any[] = []
        for (let i = 0; i < submissionsToProcess.length; i += CONCURRENCY) {
            const batch = submissionsToProcess.slice(i, i + CONCURRENCY)
            // console.log(`[CRON] Processing batch ${Math.floor(i / CONCURRENCY) + 1}/${Math.ceil(submissionsToProcess.length / CONCURRENCY)}`)

            const batchResults = await processBatch(batch as SubmissionToUpdate[])
            results.push(...batchResults)

            // Delay between batches to avoid rate limiting
            if (i + CONCURRENCY < submissionsToProcess.length) {
                await delay(DELAY_MS)
            }
        }

        // Calculate stats
        const successCount = results.filter(r => r.success).length
        const updatedCount = results.filter(r => r.viewsUpdated).length
        const errorCount = results.filter(r => !r.success).length
        const totalViewsGained = results
            .filter(r => r.viewsUpdated && r.newViews)
            .reduce((sum, r) => sum + (r.newViews || 0), 0)
        const totalRevenueGained = results
            .filter(r => r.revenueDiff && r.revenueDiff > 0)
            .reduce((sum, r) => sum + (r.revenueDiff || 0), 0)

        const durationMs = Date.now() - startTime

        // console.log(`[CRON] Completed in ${durationMs}ms`)
        // console.log(`[CRON] Processed: ${successCount}, Updated: ${updatedCount}, Errors: ${errorCount}`)
        // console.log(`[CRON] New revenue: €${totalRevenueGained.toFixed(2)}`)

        return NextResponse.json({
            success: true,
            stats: {
                totalSubmissions: approvedSubmissions.length,
                processed: submissionsToProcess.length,
                skippedAtCap: approvedSubmissions.length - submissionsToProcess.length,
                updated: updatedCount,
                errors: errorCount,
                totalViewsGained,
                totalRevenueGained: totalRevenueGained.toFixed(2)
            },
            durationMs
        })

    } catch (error) {
        console.error('[CRON] Fatal error:', error)
        return NextResponse.json({
            success: false,
            error: 'Internal Server Error',
            durationMs: Date.now() - startTime
        }, { status: 500 })
    }
}
