import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { db } from '@/lib/db'
import { campaigns, globalSettings } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import PDFDocument from 'pdfkit'
import { generatePaymentReference, type Currency, type BankDetailsEUR, type BankDetailsUSD } from '@/config/bank-details'

export async function GET(request: NextRequest) {
    try {
        const session = await auth()
        if (!session || session.user.role !== 'studio') {
            return new NextResponse('Unauthorized', { status: 401 })
        }

        const { searchParams } = new URL(request.url)
        const campaignId = searchParams.get('campaign')
        const currency = (searchParams.get('currency') as Currency) || 'EUR'

        if (!campaignId) {
            return new NextResponse('Campaign ID required', { status: 400 })
        }

        // Fetch campaign
        const campaign = await db.query.campaigns.findFirst({
            where: eq(campaigns.id, campaignId),
            with: { studio: true }
        })

        if (!campaign) {
            return new NextResponse('Campaign not found', { status: 404 })
        }

        // Verify ownership
        if (campaign.studioId !== session.user.id) {
            return new NextResponse('Forbidden', { status: 403 })
        }

        // Fetch bank details from database
        const settings = await db.query.globalSettings.findFirst()
        if (!settings) {
            return new NextResponse('Bank details not configured', { status: 500 })
        }

        const bankDetailsEUR: BankDetailsEUR = {
            beneficiary: settings.eurBeneficiary || '',
            iban: settings.eurIban || '',
            bic: settings.eurBic || '',
            bankName: settings.eurBankName || '',
            bankAddress: settings.eurBankAddress || ''
        }

        const bankDetailsUSD: BankDetailsUSD = {
            beneficiary: settings.usdBeneficiary || '',
            accountNumber: settings.usdAccountNumber || '',
            routingNumber: settings.usdRoutingNumber || '',
            bic: settings.usdBic || '',
            accountType: settings.usdAccountType || '',
            bankName: settings.usdBankName || '',
            bankAddress: settings.usdBankAddress || ''
        }

        const reference = generatePaymentReference(campaignId)
        const bankDetails = currency === 'EUR' ? bankDetailsEUR : bankDetailsUSD
        const amount = parseFloat(campaign.budget)
        const today = new Date()

        // Generate PDF
        const doc = new PDFDocument({ margin: 50, size: 'A4' })
        const chunks: Buffer[] = []

        doc.on('data', (chunk) => chunks.push(chunk))

        // Header
        doc.fontSize(24).font('Helvetica-Bold').text('FACTURE PROFORMA', { align: 'center' })
        doc.moveDown(0.5)
        doc.fontSize(10).font('Helvetica').fillColor('#666')
            .text(`Référence: ${reference}`, { align: 'center' })
        doc.text(`Date: ${today.toLocaleDateString('fr-FR')}`, { align: 'center' })
        doc.moveDown(2)

        // Emetteur (Swiplay)
        doc.fillColor('#000').font('Helvetica-Bold').fontSize(12).text('ÉMETTEUR')
        doc.font('Helvetica').fontSize(10)
        doc.text('SWIPLAY')
        doc.text(bankDetails.beneficiary)
        doc.text('contact@swiplay.com')
        doc.moveDown(1.5)

        // Client
        doc.font('Helvetica-Bold').fontSize(12).text('DESTINATAIRE')
        doc.font('Helvetica').fontSize(10)
        doc.text(campaign.studio?.company || campaign.studio?.name || 'Studio')
        doc.text(campaign.studio?.email || '')
        doc.moveDown(2)

        // Separator
        doc.moveTo(50, doc.y).lineTo(545, doc.y).stroke('#ddd')
        doc.moveDown(1)

        // Table Header
        doc.font('Helvetica-Bold').fontSize(10)
        const tableTop = doc.y
        doc.text('Description', 50, tableTop, { width: 300 })
        doc.text('Quantité', 350, tableTop, { width: 80, align: 'center' })
        doc.text(`Montant (${currency})`, 430, tableTop, { width: 115, align: 'right' })
        doc.moveDown(0.5)
        doc.moveTo(50, doc.y).lineTo(545, doc.y).stroke('#ddd')
        doc.moveDown(0.5)

        // Line Item
        doc.font('Helvetica').fontSize(10)
        const itemY = doc.y
        doc.text(`Campagne: ${campaign.title}`, 50, itemY, { width: 300 })
        doc.text('1', 350, itemY, { width: 80, align: 'center' })
        doc.text(`${amount.toFixed(2)} ${currency === 'EUR' ? '€' : '$'}`, 430, itemY, { width: 115, align: 'right' })
        doc.moveDown(0.5)

        // Description line
        doc.fontSize(9).fillColor('#666')
        doc.text(`Budget campagne marketing - Référence ${reference}`, 50)
        doc.fillColor('#000').fontSize(10)
        doc.moveDown(1.5)

        // Separator
        doc.moveTo(50, doc.y).lineTo(545, doc.y).stroke('#ddd')
        doc.moveDown(0.5)

        // Total
        doc.font('Helvetica-Bold').fontSize(12)
        const totalY = doc.y
        doc.text('TOTAL TTC', 350, totalY, { width: 80 })
        doc.text(`${amount.toFixed(2)} ${currency === 'EUR' ? '€' : '$'}`, 430, totalY, { width: 115, align: 'right' })
        doc.moveDown(3)

        // Bank Details Section
        doc.font('Helvetica-Bold').fontSize(14).fillColor('#7c3aed')
        doc.text('COORDONNÉES BANCAIRES')
        doc.moveDown(0.5)

        doc.font('Helvetica').fontSize(10).fillColor('#000')

        if (currency === 'EUR') {
            const eurDetails = bankDetails as BankDetailsEUR
            doc.text(`Bénéficiaire: ${eurDetails.beneficiary}`)
            doc.text(`IBAN: ${eurDetails.iban}`)
            doc.text(`BIC/SWIFT: ${eurDetails.bic}`)
            doc.text(`Banque: ${eurDetails.bankName}`)
            doc.text(`Adresse: ${eurDetails.bankAddress}`)
        } else {
            const usdDetails = bankDetails as BankDetailsUSD
            doc.text(`Beneficiary: ${usdDetails.beneficiary}`)
            doc.text(`Account Number: ${usdDetails.accountNumber}`)
            doc.text(`Routing Number: ${usdDetails.routingNumber}`)
            doc.text(`Account Type: ${usdDetails.accountType}`)
            doc.text(`Bank: ${usdDetails.bankName}`)
            doc.text(`Address: ${usdDetails.bankAddress}`)
        }

        doc.moveDown(1.5)

        // Payment Reference Box
        doc.rect(50, doc.y, 495, 60).fillAndStroke('#fef3c7', '#f59e0b')
        doc.fillColor('#92400e').fontSize(11).font('Helvetica-Bold')
        doc.text('RÉFÉRENCE DE PAIEMENT (À INDIQUER DANS LE LIBELLÉ)', 60, doc.y - 50, { width: 475, align: 'center' })
        doc.fontSize(20).fillColor('#b45309')
        doc.text(reference, 60, doc.y + 5, { width: 475, align: 'center' })

        // Footer
        doc.fillColor('#666').fontSize(8)
        doc.text(
            'Document non contractuel - Ce document est une facture proforma et ne constitue pas une facture définitive.',
            50,
            750,
            { align: 'center', width: 495 }
        )

        doc.end()

        // Wait for PDF to be fully generated
        await new Promise<void>((resolve) => doc.on('end', resolve))

        const pdfBuffer = Buffer.concat(chunks)

        return new NextResponse(pdfBuffer, {
            headers: {
                'Content-Type': 'application/pdf',
                'Content-Disposition': `attachment; filename="Proforma_${reference}.pdf"`,
            }
        })

    } catch (error) {
        console.error('Proforma generation error:', error)
        return new NextResponse('Error generating invoice', { status: 500 })
    }
}
