import { NextRequest, NextResponse } from 'next/server'
import { requireAdmin } from '@/lib/auth-middleware'
import { db } from '@/lib/db'
import { globalSettings } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'

export async function GET(request: NextRequest) {
  const authResult = await requireAdmin(request)
  if (authResult.response) return authResult.response

  try {
    // Fetch settings (singleton)
    let settings = await db.query.globalSettings.findFirst()

    // If no settings exist, create default
    if (!settings) {
      const [newSettings] = await db.insert(globalSettings).values({
        platformFee: 10,
        minPayout: 50,
        autoApproveCreators: 0,
        requireStudioApproval: 1,
        emailNotifications: 1,
        weeklyReports: 1,
        maintenanceMode: 0
      }).returning()
      settings = newSettings
    }

    // Transform for frontend (boolean conversion)
    const formattedSettings = {
      ...settings,
      autoApproveCreators: settings.autoApproveCreators === 1,
      requireStudioApproval: settings.requireStudioApproval === 1,
      emailNotifications: settings.emailNotifications === 1,
      weeklyReports: settings.weeklyReports === 1,
      maintenanceMode: settings.maintenanceMode === 1,
    }

    return NextResponse.json({ settings: formattedSettings })
  } catch (error) {
    console.error('Error fetching settings:', error)
    return NextResponse.json(
      { error: 'Failed to fetch settings' },
      { status: 500 }
    )
  }
}

export async function PUT(request: NextRequest) {
  const authResult = await requireAdmin(request)
  if (authResult.response) return authResult.response

  try {
    const body = await request.json()

    // Fetch existing settings to get ID
    const existingSettings = await db.query.globalSettings.findFirst()

    if (!existingSettings) {
      return NextResponse.json({ error: 'Settings not initialized' }, { status: 404 })
    }

    // Update settings
    const [updatedSettings] = await db.update(globalSettings)
      .set({
        platformFee: body.platformFee,
        minPayout: body.minPayout,
        autoApproveCreators: body.autoApproveCreators ? 1 : 0,
        requireStudioApproval: body.requireStudioApproval ? 1 : 0,
        emailNotifications: body.emailNotifications ? 1 : 0,
        weeklyReports: body.weeklyReports ? 1 : 0,
        maintenanceMode: body.maintenanceMode ? 1 : 0,
        updatedAt: new Date()
      })
      .where(eq(globalSettings.id, existingSettings.id))
      .returning()

    // Transform for frontend
    const formattedSettings = {
      ...updatedSettings,
      autoApproveCreators: updatedSettings.autoApproveCreators === 1,
      requireStudioApproval: updatedSettings.requireStudioApproval === 1,
      emailNotifications: updatedSettings.emailNotifications === 1,
      weeklyReports: updatedSettings.weeklyReports === 1,
      maintenanceMode: updatedSettings.maintenanceMode === 1,
    }

    return NextResponse.json({
      success: true,
      settings: formattedSettings
    })
  } catch (error) {
    console.error('Error updating settings:', error)
    return NextResponse.json(
      { error: 'Failed to update settings' },
      { status: 500 }
    )
  }
}
