import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { users, campaigns, submissions } from '@/lib/db/schema'
import { eq, and, sql } from 'drizzle-orm'
import { requireAdmin } from '@/lib/auth-middleware'

export async function GET(request: NextRequest) {
  const authResult = await requireAdmin(request)
  if (authResult.response) return authResult.response

  try {
    // Count creators
    const creatorsCount = await db
      .select({ count: sql<number>`count(*)` })
      .from(users)
      .where(eq(users.role, 'creator'))

    // Count approved studios
    const studiosCount = await db
      .select({ count: sql<number>`count(*)` })
      .from(users)
      .where(and(
        eq(users.role, 'studio'),
        eq(users.isApproved, 1)
      ))

    // Count pending studio requests
    const pendingCount = await db
      .select({ count: sql<number>`count(*)` })
      .from(users)
      .where(and(
        eq(users.role, 'studio'),
        eq(users.isApproved, 0)
      ))

    // Count active campaigns
    const activeCampaignsCount = await db
      .select({ count: sql<number>`count(*)` })
      .from(campaigns)
      .where(eq(campaigns.status, 'active'))

    // Calculate total revenue from approved submissions
    const revenueData = await db
      .select({
        totalRevenue: sql<number>`COALESCE(SUM(CAST(${submissions.revenue} AS DECIMAL)), 0)`,
      })
      .from(submissions)
      .where(eq(submissions.status, 'approved'))

    // Calculate this month revenue
    const firstDayOfMonth = new Date(new Date().getFullYear(), new Date().getMonth(), 1)
    const monthRevenueData = await db
      .select({
        monthRevenue: sql<number>`COALESCE(SUM(CAST(${submissions.revenue} AS DECIMAL)), 0)`,
      })
      .from(submissions)
      .where(and(
        eq(submissions.status, 'approved'),
        sql`${submissions.submittedAt} >= ${firstDayOfMonth}`
      ))

    return NextResponse.json({
      totalCreators: Number(creatorsCount[0]?.count || 0),
      totalStudios: Number(studiosCount[0]?.count || 0),
      pendingRequests: Number(pendingCount[0]?.count || 0),
      activeCampaigns: Number(activeCampaignsCount[0]?.count || 0),
      totalRevenue: Number(revenueData[0]?.totalRevenue || 0),
      thisMonthRevenue: Number(monthRevenueData[0]?.monthRevenue || 0)
    })
  } catch (error) {
    console.error('Error fetching stats:', error)
    return NextResponse.json(
      { error: 'Failed to fetch stats' },
      { status: 500 }
    )
  }
}
