import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { campaignAssets, campaigns } from '@/lib/db/schema'
import { googleDriveService } from '@/lib/google-drive'
import { eq } from 'drizzle-orm'

export async function POST(req: NextRequest) {
    try {
        const formData = await req.formData()
        const file = formData.get('file') as File
        const campaignId = formData.get('campaignId') as string
        const fileType = formData.get('fileType') as string || 'other'

        if (!file || !campaignId) {
            return NextResponse.json(
                { error: 'File and campaignId are required' },
                { status: 400 }
            )
        }

        // Get campaign and ensure asset folder exists
        const campaign = await db.query.campaigns.findFirst({
            where: eq(campaigns.id, campaignId)
        })

        if (!campaign) {
            return NextResponse.json(
                { error: 'Campaign not found' },
                { status: 404 }
            )
        }

        // Create campaign folder in Google Drive if it doesn't exist
        let folderId = campaign.assetFolderId
        if (!folderId) {
            // Create folder with campaign title
            const folderResult = await googleDriveService.createFolder(
                `Campaign_${campaign.title.replace(/[^a-z0-9]/gi, '_')}_${campaignId.slice(0, 8)}`
            )

            if (!folderResult.success || !folderResult.folderId) {
                return NextResponse.json(
                    { error: 'Failed to create asset folder' },
                    { status: 500 }
                )
            }

            folderId = folderResult.folderId

            // Update campaign with folder ID
            await db.update(campaigns)
                .set({ assetFolderId: folderId })
                .where(eq(campaigns.id, campaignId))
        }

        // Convert file to buffer
        const arrayBuffer = await file.arrayBuffer()
        const buffer = Buffer.from(arrayBuffer)

        // Upload to Google Drive
        const uploadResult = await googleDriveService.uploadFile(
            buffer,
            file.name,
            file.type,
            folderId
        )

        if (!uploadResult.success || !uploadResult.fileId) {
            return NextResponse.json(
                { error: uploadResult.error || 'Upload failed' },
                { status: 500 }
            )
        }

        // Get direct download URL
        const directUrl = `https://drive.google.com/uc?export=download&id=${uploadResult.fileId}`

        // Save metadata to database
        const [asset] = await db.insert(campaignAssets).values({
            campaignId,
            fileId: uploadResult.fileId,
            fileName: file.name,
            originalName: file.name,
            mimeType: file.type,
            fileType,
            driveUrl: directUrl,
            size: file.size,
        }).returning()

        return NextResponse.json({
            success: true,
            asset: {
                id: asset.id,
                fileName: asset.fileName,
                fileType: asset.fileType,
                driveUrl: asset.driveUrl,
                size: asset.size,
                uploadedAt: asset.uploadedAt,
            }
        })

    } catch (error: any) {
        console.error('Asset upload error:', error)
        return NextResponse.json(
            { error: error.message || 'Internal server error' },
            { status: 500 }
        )
    }
}
