import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { userSettings, users } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { auth } from '@/auth'
import logger from '@/lib/logger'

// GET - Fetch studio settings
export async function GET(request: NextRequest) {
    try {
        const session = await auth()

        if (!session || !session.user) {
            return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
        }

        const userId = session.user.id

        // Get user settings
        let settings = await db.query.userSettings.findFirst({
            where: eq(userSettings.userId, userId)
        })

        // Get user profile (for company info)
        const user = await db.query.users.findFirst({
            where: eq(users.id, userId)
        })

        // If no settings exist, create default ones
        if (!settings) {
            const newSettings = await db.insert(userSettings).values({
                userId: userId,
                emailNotifications: 1,
                newCampaignsNotifications: 1,
                submissionStatusNotifications: 1,
                paymentNotifications: 1,
                marketingEmails: 0,
                profileVisible: 1,
                statsVisible: 0,
            }).returning()

            settings = newSettings[0]
        }

        return NextResponse.json({
            profile: {
                name: user?.name || '',
                company: user?.company || '',
                email: user?.email || '',
            },
            notifications: {
                email: settings.emailNotifications === 1,
                campaigns: settings.newCampaignsNotifications === 1,
                submissions: settings.submissionStatusNotifications === 1,
            }
        })
    } catch (error) {
        logger.error({ err: error }, 'Get settings error')
        return NextResponse.json(
            { error: 'Erreur lors de la récupération des paramètres' },
            { status: 500 }
        )
    }
}

// PUT - Update studio settings
export async function PUT(request: NextRequest) {
    try {
        const session = await auth()

        if (!session || !session.user) {
            return NextResponse.json({ error: 'Non autorisé' }, { status: 401 })
        }

        const userId = session.user.id

        const data = await request.json()
        const { notifications, profile } = data

        // Update User Settings
        const existingSettings = await db.query.userSettings.findFirst({
            where: eq(userSettings.userId, userId)
        })

        if (existingSettings) {
            await db.update(userSettings)
                .set({
                    emailNotifications: notifications?.email ? 1 : 0,
                    newCampaignsNotifications: notifications?.campaigns ? 1 : 0,
                    submissionStatusNotifications: notifications?.submissions ? 1 : 0,
                    updatedAt: new Date()
                })
                .where(eq(userSettings.userId, userId))
        } else {
            await db.insert(userSettings).values({
                userId: userId,
                emailNotifications: notifications?.email ? 1 : 0,
                newCampaignsNotifications: notifications?.campaigns ? 1 : 0,
                submissionStatusNotifications: notifications?.submissions ? 1 : 0,
            })
        }

        // Update User Profile (Company Info)
        if (profile) {
            await db.update(users)
                .set({
                    name: profile.name,
                    company: profile.company,
                    updatedAt: new Date()
                })
                .where(eq(users.id, userId))
        }

        return NextResponse.json({
            message: 'Paramètres mis à jour avec succès'
        })
    } catch (error) {
        logger.error({ err: error }, 'Update settings error')
        return NextResponse.json(
            { error: 'Erreur lors de la mise à jour des paramètres' },
            { status: 500 }
        )
    }
}
