import { NextResponse } from 'next/server'
import { auth } from '@/auth'
import { db } from '@/lib/db'
import { submissions, campaigns, users } from '@/lib/db/schema'
import { eq, desc, inArray } from 'drizzle-orm'

export async function GET() {
    try {
        const session = await auth()
        if (!session?.user || session.user.role !== 'studio') {
            return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
        }

        // 1. Get all campaigns owned by this studio
        const studioCampaigns = await db.query.campaigns.findMany({
            where: eq(campaigns.studioId, session.user.id),
            columns: { id: true }
        })

        const campaignIds = studioCampaigns.map(c => c.id)

        if (campaignIds.length === 0) {
            return NextResponse.json([])
        }

        // 2. Get all submissions for these campaigns
        const studioSubmissions = await db.query.submissions.findMany({
            where: inArray(submissions.campaignId, campaignIds),
            with: {
                campaign: {
                    columns: {
                        id: true,
                        title: true,
                        pricePerView: true,
                        coverImage: true,
                        icon: true
                    }
                },
                creator: {
                    columns: {
                        id: true,
                        name: true,
                        email: true,
                        image: true
                    }
                }
            },
            orderBy: [desc(submissions.submittedAt)]
        })

        return NextResponse.json(studioSubmissions)

    } catch (error) {
        console.error('Studio submissions error:', error)
        return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 })
    }
}
