import { NextRequest, NextResponse } from 'next/server'
import { authenticateRequest, requireRole } from '@/lib/auth-middleware'
import { googleDriveService } from '@/lib/google-drive'
import { handleApiError, createApiResponse } from '@/lib/api-response'
import { db } from '@/lib/db'
import { campaigns } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { campaignAssets } from '@/lib/db/schema'
import { sql } from 'drizzle-orm'

export async function POST(request: NextRequest) {
  try {
    const authResult = await authenticateRequest(request)
    if (!authResult.success) {
      return NextResponse.json({ error: authResult.error }, { status: 401 })
    }

    const roleCheck = requireRole(authResult.role, 'studio')
    if (!roleCheck.allowed) {
      return NextResponse.json({ error: roleCheck.error }, { status: 403 })
    }

    const formData = await request.formData()
    const file = formData.get('file') as File
    const campaignId = formData.get('campaignId') as string

    if (!file) {
      return NextResponse.json(
        { error: 'Fichier requis' },
        { status: 400 }
      )
    }

    if (!campaignId) {
      return NextResponse.json(
        { error: 'ID de campagne requis' },
        { status: 400 }
      )
    }

    // Vérifier que la campagne appartient au studio
    const campaign = await db.query.campaigns.findFirst({
      where: eq(campaigns.id, campaignId)
    })

    if (!campaign) {
      return NextResponse.json(
        { error: 'Campagne non trouvée' },
        { status: 404 }
      )
    }

    if (campaign.studioId !== authResult.userId) {
      return NextResponse.json(
        { error: 'Accès non autorisé à cette campagne' },
        { status: 403 }
      )
    }

    // Vérifier la taille du fichier (limite 5GB par campagne)
    const MAX_FILE_SIZE = 5 * 1024 * 1024 * 1024 // 5GB
    if (file.size > MAX_FILE_SIZE) {
      return NextResponse.json(
        { error: 'Fichier trop volumineux (max 5GB)' },
        { status: 400 }
      )
    }

    // Vérifier l'espace total déjà utilisé par la campagne
    const existingSizeResult: any = await db.query.campaign_assets.findMany({
      where: eq(campaignAssets.campaignId, campaignId),
      columns: { size: true }
    })

    const totalExisting = existingSizeResult.reduce((acc: number, r: any) => acc + (Number(r.size) || 0), 0)
    if ((totalExisting + file.size) > MAX_FILE_SIZE) {
      return NextResponse.json(
        { error: 'Limite de stockage de la campagne atteinte (5GB)' },
        { status: 400 }
      )
    }

    // Types de fichiers autorisés
    const allowedTypes = [
      'image/jpeg', 'image/png', 'image/gif', 'image/webp',
      'video/mp4', 'video/webm', 'video/avi',
      'audio/mp3', 'audio/wav', 'audio/mpeg'
    ]

    if (!allowedTypes.includes(file.type)) {
      return NextResponse.json(
        { error: 'Type de fichier non autorisé' },
        { status: 400 }
      )
    }

    // Convertir le fichier en buffer
    const arrayBuffer = await file.arrayBuffer()
    const buffer = Buffer.from(arrayBuffer)

    // Générer un nom de fichier unique
    const fileExtension = file.name.split('.').pop()
    const uniqueFileName = `${campaignId}_${Date.now()}_${Math.random().toString(36).substring(2)}.${fileExtension}`

    // Uploader vers Google Drive
    const uploadResult = await googleDriveService.uploadFile(
      buffer,
      uniqueFileName,
      file.type
    )

    if (!uploadResult.success) {
      return NextResponse.json(
        { error: uploadResult.error || 'Erreur lors de l\'upload' },
        { status: 500 }
      )
    }

    // Enregistrer les métadonnées du fichier en base
    try {
      await db.insert(campaignAssets).values({
        campaignId: campaignId,
        fileId: uploadResult.fileId!,
        fileName: uniqueFileName,
        originalName: file.name,
        mimeType: file.type,
        size: String(buffer.length),
        uploadedAt: new Date()
      })
    } catch (e: any) {
      console.error('Erreur enregistrement campaign asset:', e)
      // Rollback: attempt to delete the file from Drive to avoid orphaned files
      if (uploadResult.fileId) {
        await googleDriveService.deleteFile(uploadResult.fileId)
      }
      return NextResponse.json(
        { error: 'Erreur lors de l\'enregistrement du fichier' },
        { status: 500 }
      )
    }

    return createApiResponse({
      fileId: uploadResult.fileId,
      url: uploadResult.url,
      fileName: uniqueFileName,
      originalName: file.name,
      size: file.size,
      type: file.type,
      uploadedAt: new Date().toISOString()
    })

  } catch (error) {
    return handleApiError(error)
  }
}

// Route pour supprimer un fichier
export async function DELETE(request: NextRequest) {
  try {
    const authResult = await authenticateRequest(request)
    if (!authResult.success) {
      return NextResponse.json({ error: authResult.error }, { status: 401 })
    }

    const roleCheck = requireRole(authResult.role, 'studio')
    if (!roleCheck.allowed) {
      return NextResponse.json({ error: roleCheck.error }, { status: 403 })
    }

    const { searchParams } = new URL(request.url)
    const fileId = searchParams.get('fileId')

    if (!fileId) {
      return NextResponse.json(
        { error: 'ID de fichier requis' },
        { status: 400 }
      )
    }

    const deleteResult = await googleDriveService.deleteFile(fileId)

    if (!deleteResult.success) {
      return NextResponse.json(
        { error: deleteResult.error || 'Erreur lors de la suppression' },
        { status: 500 }
      )
    }

    return createApiResponse({ message: 'Fichier supprimé avec succès' })

  } catch (error) {
    return handleApiError(error)
  }
}