import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { submissions } from '@/lib/db/schema'
import { eq, inArray } from 'drizzle-orm'
import { authenticateRequest, requireRole } from '@/lib/auth-middleware'
import { sendDiscordNotification } from '@/lib/discord'

// Bulk approve/reject submissions
export async function POST(request: NextRequest) {
    try {
        const authResult = await authenticateRequest(request)
        if (!authResult.success) {
            return NextResponse.json({ error: authResult.error }, { status: 401 })
        }

        const roleCheck = requireRole(authResult.role, 'studio')
        if (!roleCheck.allowed) {
            return NextResponse.json({ error: roleCheck.error }, { status: 403 })
        }

        const { ids, action } = await request.json()

        if (!ids || !Array.isArray(ids) || ids.length === 0) {
            return NextResponse.json({ error: 'IDs requis' }, { status: 400 })
        }

        if (!['approve', 'reject'].includes(action)) {
            return NextResponse.json({ error: 'Action invalide (approve/reject)' }, { status: 400 })
        }

        const newStatus = action === 'approve' ? 'approved' : 'rejected'

        // Update all submissions
        await db.update(submissions)
            .set({ status: newStatus })
            .where(inArray(submissions.id, ids))

        // Discord notification
        await sendDiscordNotification(
            action === 'approve' ? '✅ Soumissions Approuvées en Masse' : '❌ Soumissions Rejetées en Masse',
            `${ids.length} soumission(s) ${action === 'approve' ? 'approuvée(s)' : 'rejetée(s)'} par le studio`,
            action === 'approve' ? 'success' : 'warning',
            [{ name: 'Nombre', value: ids.length.toString(), inline: true }]
        )

        return NextResponse.json({
            success: true,
            action,
            count: ids.length
        })

    } catch (error) {
        console.error('Bulk submission error:', error)
        return NextResponse.json({ error: 'Erreur serveur' }, { status: 500 })
    }
}

