import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'
import { creatorProfiles } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import { authenticateRequest, requireRole } from '@/lib/auth-middleware'
import { z } from 'zod'

// Configuration finale des créateurs (basée sur v1)
const CREATORS_CONFIG = {
  eyzuro: {
    name: "Eyzuro",
    platforms: {
      instagram: { username: "eyzuro" },
      tiktok: { username: "eyzuro" },
      youtube: { channelId: "@Eyzuro" }
    }
  },
  rifflop: {
    name: "Rifflop",
    platforms: {
      youtube: { channelId: "@RiffleStudio" },
      tiktok: { username: "rifflop" },
      instagram: { username: "rifflop_" }
    }
  },
  kouma: {
    name: "Kouma",
    platforms: {
      instagram: { username: "koumaoff" },
      tiktok: { username: "koumaoff_" },
      youtube: { channelId: "@KoumaOff_" }
    }
  },
  moura: {
    name: "Moura",
    platforms: {
      instagram: { username: "mouracreateur" },
      tiktok: { username: "mouracreateur" },
      youtube: { channelId: "@mouracreateur" }
    }
  }
};

// Headers optimisés pour le scraping
const HEADERS = {
  youtube: {
    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
    'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
    'Accept-Language': 'en-US,en;q=0.9',
    'Accept-Encoding': 'gzip, deflate, br',
    'DNT': '1',
    'Connection': 'keep-alive',
    'Upgrade-Insecure-Requests': '1'
  },
  common: {
    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
  }
};

// Fonction pour parser les nombres avec unités
function parseNumber(text: string): number {
  if (!text) return 0;

  const cleaned = text.replace(/[,\s]/g, '').toUpperCase();
  const number = parseFloat(cleaned);

  if (cleaned.includes('K')) {
    return Math.round(number * 1000);
  } else if (cleaned.includes('M')) {
    return Math.round(number * 1000000);
  } else if (cleaned.includes('B')) {
    return Math.round(number * 1000000000);
  }

  return Math.round(number) || 0;
}

// Fonction pour formater les nombres
function formatNumber(num: number): string {
  if (num >= 1000000) {
    return (num / 1000000).toFixed(1) + 'M'
  } else if (num >= 1000) {
    return (num / 1000).toFixed(1) + 'k'
  }
  return num.toString()
}

// Instagram tracking (méthode embed réaliste)
async function getInstagramFollowers(username: string) {
  try {
    const embedUrl = `https://www.instagram.com/${username}/embed/`

    const response = await fetch(embedUrl, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,*/*;q=0.8',
        'Accept-Language': 'en-US,en;q=0.5',
        'Accept-Encoding': 'gzip, deflate, br',
        'Referer': 'https://www.instagram.com/',
        'Connection': 'keep-alive',
        'Upgrade-Insecure-Requests': '1',
        'Sec-Fetch-Dest': 'iframe',
        'Sec-Fetch-Mode': 'navigate',
        'Sec-Fetch-Site': 'same-origin'
      },
      signal: AbortSignal.timeout(15000)
    })

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`)
    }

    const html = await response.text()

    // Patterns pour extraire les followers
    const patterns = [
      /\\"followers_count\\":(\d+)/g,
      /\\"edge_followed_by\\":\s*\{\s*\\"count\\":(\d+)\s*\}/g
    ]

    for (const pattern of patterns) {
      const matches = [...html.matchAll(pattern)]
      if (matches.length > 0) {
        const followers = matches.map(match => parseInt(match[1]))
        const maxFollowers = Math.max(...followers)

        if (maxFollowers > 0) {
          return {
            success: true,
            followers: maxFollowers,
            formatted: formatNumber(maxFollowers),
            method: 'instagram_embed',
            lastChecked: new Date().toISOString()
          }
        }
      }
    }

    return {
      success: false,
      error: 'No followers count found',
      lastChecked: new Date().toISOString()
    }
  } catch (error) {
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error',
      lastChecked: new Date().toISOString()
    }
  }
}

// YouTube tracking (scraping HTML temps réel)
async function getYouTubeSubscribers(channelHandle: string) {
  try {
    // Nettoyer le handle (enlever @ si présent)
    const cleanHandle = channelHandle.replace('@', '')

    // Essayer d'abord avec le format @handle
    let url = `https://www.youtube.com/@${cleanHandle}`

    let response = await fetch(url, {
      headers: HEADERS.youtube,
      signal: AbortSignal.timeout(15000)
    })

    // Si ça ne marche pas, essayer avec /c/ ou /channel/
    if (!response.ok) {
      // Essayer /c/handle
      url = `https://www.youtube.com/c/${cleanHandle}`
      response = await fetch(url, {
        headers: HEADERS.youtube,
        signal: AbortSignal.timeout(15000)
      })

      if (!response.ok) {
        throw new Error(`YouTube page not found for ${channelHandle}`)
      }
    }

    const html = await response.text()

    // Patterns pour extraire les abonnés
    const patterns = [
      /"subscriberCountText":\s*\{\s*"runs":\s*\[\s*\{\s*"text":\s*"([^"]+)"\s*\}\s*\]\s*\}/g,
      /"subscriberCountText":\s*\{\s*"simpleText":\s*"([^"]+)"\s*\}/g,
      /subscriberCountText[^}]*simpleText[^"]*"([^"]+)"/g,
      /([0-9,]+(?:\.[0-9]+)?[KMB]?)\s+subscribers?/i
    ]

    for (const pattern of patterns) {
      const matches = [...html.matchAll(pattern)]
      for (const match of matches) {
        const subscriberText = match[1] || match[0]
        const subscribers = parseNumber(subscriberText)

        if (subscribers > 0) {
          return {
            success: true,
            subscribers: subscribers,
            formatted: formatNumber(subscribers),
            method: 'youtube_scraping',
            channelHandle: channelHandle,
            url: url,
            lastChecked: new Date().toISOString()
          }
        }
      }
    }

    return {
      success: false,
      error: 'No subscriber count found',
      lastChecked: new Date().toISOString()
    }
  } catch (error) {
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error',
      lastChecked: new Date().toISOString()
    }
  }
}

// TikTok tracking (scraping HTML temps réel)
async function getTikTokFollowers(username: string) {
  try {
    const url = `https://www.tiktok.com/@${username}`

    const response = await fetch(url, {
      headers: {
        ...HEADERS.common,
        'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
        'Accept-Language': 'en-US,en;q=0.9',
        'Accept-Encoding': 'gzip, deflate, br',
        'DNT': '1',
        'Connection': 'keep-alive',
        'Upgrade-Insecure-Requests': '1'
      },
      signal: AbortSignal.timeout(15000)
    })

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`)
    }

    const html = await response.text()

    // Patterns pour extraire les followers
    const patterns = [
      /"followerCount":\s*(\d+)/g,
      /"followers":\s*\{\s*"count":\s*(\d+)\s*\}/g,
      /([0-9,]+(?:\.[0-9]+)?[KMB]?)\s+Followers?/i,
      /\\"followerCount\\":\\"(\d+)\\"/g,
      /\\"stats\\":\s*\{[^}]*\\"followerCount\\":\s*(\d+)/g
    ]

    for (const pattern of patterns) {
      const matches = [...html.matchAll(pattern)]
      for (const match of matches) {
        const followerText = match[1] || match[0]
        const followers = parseNumber(followerText)

        if (followers > 0) {
          return {
            success: true,
            followers: followers,
            formatted: formatNumber(followers),
            method: 'tiktok_scraping',
            username: username,
            url: url,
            lastChecked: new Date().toISOString()
          }
        }
      }
    }

    return {
      success: false,
      error: 'No follower count found',
      lastChecked: new Date().toISOString()
    }
  } catch (error) {
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error',
      lastChecked: new Date().toISOString()
    }
  }
}

// Fonction principale pour récupérer tous les abonnés
async function fetchAllSubscribers() {
  const results: any = {}
  let successfulFetches = 0
  let failedFetches = 0

  for (const [creatorKey, creatorData] of Object.entries(CREATORS_CONFIG)) {
    results[creatorKey] = {
      name: creatorData.name,
      platforms: {}
    }

    // Instagram
    if (creatorData.platforms.instagram) {
      try {
        const instagramResult = await getInstagramFollowers(creatorData.platforms.instagram.username)
        results[creatorKey].platforms.instagram = instagramResult
        if (instagramResult.success) {
          successfulFetches++
        } else {
          failedFetches++
        }
      } catch (error) {
        results[creatorKey].platforms.instagram = {
          success: false,
          error: 'Failed to fetch',
          lastChecked: new Date().toISOString()
        }
        failedFetches++
      }
    }

    // YouTube
    if (creatorData.platforms.youtube) {
      try {
        const youtubeResult = await getYouTubeSubscribers(creatorData.platforms.youtube.channelId)
        results[creatorKey].platforms.youtube = youtubeResult
        if (youtubeResult.success) {
          successfulFetches++
        } else {
          failedFetches++
        }
      } catch (error) {
        results[creatorKey].platforms.youtube = {
          success: false,
          error: 'Failed to fetch',
          lastChecked: new Date().toISOString()
        }
        failedFetches++
      }
    }

    // TikTok
    if (creatorData.platforms.tiktok) {
      try {
        const tiktokResult = await getTikTokFollowers(creatorData.platforms.tiktok.username)
        results[creatorKey].platforms.tiktok = tiktokResult
        if (tiktokResult.success) {
          successfulFetches++
        } else {
          failedFetches++
        }
      } catch (error) {
        results[creatorKey].platforms.tiktok = {
          success: false,
          error: 'Failed to fetch',
          lastChecked: new Date().toISOString()
        }
        failedFetches++
      }
    }
  }

  return {
    creators: results,
    stats: {
      total_creators: Object.keys(CREATORS_CONFIG).length,
      platforms: ['instagram', 'youtube', 'tiktok'],
      successful_fetches: successfulFetches,
      failed_fetches: failedFetches
    }
  }
}

// GET /api/tracking/subscribers - Récupérer les données d'abonnés
export async function GET(request: NextRequest) {
  try {
    const authResult = await authenticateRequest(request)
    if (!authResult.success) {
      return NextResponse.json({ error: authResult.error }, { status: 401 })
    }

    // Récupération temps réel des données
    const trackingData = await fetchAllSubscribers()

    return NextResponse.json({
      success: true,
      api_version: '3.0',
      data_source: 'swiplay-tracker-v3',
      cache_info: {
        last_update: new Date().toISOString(),
        update_frequency: 'realtime',
        next_update: new Date(Date.now() + 60 * 60 * 1000).toISOString()
      },
      creators: trackingData.creators,
      stats: trackingData.stats
    })
  } catch (error) {
    console.error('Tracking API error:', error)
    return NextResponse.json(
      { error: 'Failed to fetch tracking data' },
      { status: 500 }
    )
  }
}

// POST /api/tracking/update - Forcer la mise à jour des données
export async function POST(request: NextRequest) {
  try {
    const authResult = await authenticateRequest(request)
    if (!authResult.success) {
      return NextResponse.json({ error: authResult.error }, { status: 401 })
    }

    const roleCheck = requireRole(authResult.role, 'admin')
    if (!roleCheck.allowed) {
      return NextResponse.json({ error: roleCheck.error }, { status: 403 })
    }

    // Forcer la mise à jour temps réel
    const trackingData = await fetchAllSubscribers()

    return NextResponse.json({
      success: true,
      message: 'Tracking update completed',
      timestamp: new Date().toISOString(),
      data: trackingData
    })
  } catch (error) {
    console.error('Tracking update error:', error)
    return NextResponse.json(
      { error: 'Failed to update tracking data' },
      { status: 500 }
    )
  }
}