import { NextRequest, NextResponse } from "next/server"
import { auth } from "@/auth"
import { saveFile } from "@/lib/storage"

import { rateLimiter } from '@/lib/rate-limit'

export async function POST(request: NextRequest) {
    const ip = request.headers.get('x-forwarded-for') || 'unknown'
    const success = rateLimiter.check(10, ip) // 10 uploads per minute

    if (!success) {
        return NextResponse.json({ error: 'Too many requests' }, { status: 429 })
    }

    try {
        const session = await auth()
        if (!session || session.user.role !== 'studio') {
            return NextResponse.json({ error: "Unauthorized" }, { status: 401 })
        }

        const formData = await request.formData()
        const file = formData.get("file") as File
        const campaignId = (formData.get("campaignId") as string) || 'general'

        if (!file) {
            return NextResponse.json({ error: "Missing file" }, { status: 400 })
        }

        const result = await saveFile(file, campaignId, file.name)

        if (result.error) {
            return NextResponse.json({ error: result.error }, { status: 400 })
        }

        return NextResponse.json({ success: true, path: result.path })
    } catch (error) {
        console.error("Upload API Error:", error)
        return NextResponse.json({ error: error instanceof Error ? error.message : "Internal Server Error" }, { status: 500 })
    }
}
