"use client"

import { useState, useEffect } from 'react'
import { toast } from 'sonner'

export type Asset = {
  id?: string
  type: 'file' | 'link'
  name: string
  size?: number
  url?: string
  file?: File
}
export type AssetUploadProps = {
  value?: Asset[]
  onChange?: (assets: Asset[]) => void
  maxTotalBytes?: number
  campaignId?: string
} & React.HTMLAttributes<HTMLDivElement>

export function AssetUpload({
  value = [],
  onChange = () => { },
  maxTotalBytes = 50 * 1024 * 1024, // 50MB
  campaignId,
  ...rest
}: AssetUploadProps) {
  const [assets, setAssets] = useState<Asset[]>(value)
  const [error, setError] = useState<string | null>(null)
  const [uploading, setUploading] = useState(false)

  useEffect(() => {
    if (campaignId) {
      loadAssets()
    } else {
      setAssets(value)
    }
  }, [value, campaignId])

  useEffect(() => {
    if (!campaignId) {
      onChange(assets)
    }
  }, [assets, onChange, campaignId])

  const loadAssets = async () => {
    try {
      const res = await fetch(`/api/studio/campaigns/${campaignId}/assets`, {
        headers: { 'Authorization': `Bearer ${localStorage.getItem('token')}` }
      })
      const data = await res.json()
      if (data.assets) {
        setAssets(data.assets.map((a: any) => ({
          id: a.id,
          name: a.fileName,
          type: a.fileType,
          url: a.driveUrl,
          size: a.size
        })))
      }
    } catch (e) {
      console.error('Failed to load assets', e)
    }
  }

  const totalBytes = assets.reduce((sum, a) => sum + (a.size || 0), 0)

  const handleFiles = async (files: FileList | null) => {
    if (!files) return

    if (campaignId) {
      setUploading(true)
      try {
        for (let i = 0; i < files.length; i++) {
          const f = files[i]
          const formData = new FormData()
          formData.append('file', f)

          const res = await fetch(`/api/studio/campaigns/${campaignId}/assets`, {
            method: 'POST',
            headers: { 'Authorization': `Bearer ${localStorage.getItem('token')}` },
            body: formData
          })

          if (!res.ok) throw new Error('Upload failed')
        }
        await loadAssets()
        toast.success('Fichiers uploadés')
      } catch (e) {
        toast.error('Erreur lors de l\'upload')
      } finally {
        setUploading(false)
      }
    } else {
      const newAssets: Asset[] = []
      for (let i = 0; i < files.length; i++) {
        const f = files[i]
        newAssets.push({ type: 'file', name: f.name, size: f.size, file: f })
      }

      if (totalBytes + newAssets.reduce((s, a) => s + (a.size || 0), 0) > maxTotalBytes) {
        setError('Taille totale dépassée (50 MB max). Pour des fichiers plus volumineux, contactez-nous ou envoyez un lien Drive.')
        return
      }

      setError(null)
      setAssets((prev) => [...prev, ...newAssets])
    }
  }

  const handleAddLink = async (e: React.FormEvent) => {
    e.preventDefault()
    const form = e.target as HTMLFormElement
    const input = form.elements.namedItem('link') as HTMLInputElement
    const url = input.value.trim()
    if (!url) return

    if (campaignId) {
      setUploading(true)
      try {
        const res = await fetch(`/api/studio/campaigns/${campaignId}/assets`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${localStorage.getItem('token')}`
          },
          body: JSON.stringify({ name: url, url })
        })
        if (!res.ok) throw new Error('Save failed')
        await loadAssets()
        toast.success('Lien ajouté')
        input.value = ''
      } catch (e) {
        toast.error('Erreur lors de l\'ajout du lien')
      } finally {
        setUploading(false)
      }
    } else {
      setAssets((prev) => [...prev, { type: 'link', name: url, url }])
      input.value = ''
    }
  }

  const handleRemove = async (index: number) => {
    if (campaignId) {
      const asset = assets[index]
      if (!asset.id) return

      try {
        const res = await fetch(`/api/studio/campaigns/${campaignId}/assets?assetId=${asset.id}`, {
          method: 'DELETE',
          headers: { 'Authorization': `Bearer ${localStorage.getItem('token')}` }
        })
        if (!res.ok) throw new Error('Delete failed')
        await loadAssets()
        toast.success('Asset supprimé')
      } catch (e) {
        toast.error('Erreur lors de la suppression')
      }
    } else {
      setAssets((prev) => prev.filter((_, i) => i !== index))
    }
  }

  return (
    <div {...rest} className={`space-y-3 ${rest.className || ''}`}>
      <div>
        <label className="block text-sm font-medium mb-1">Fichiers (max total 50 MB)</label>
        <input
          type="file"
          multiple
          onChange={(e) => handleFiles(e.target.files)}
          className="text-sm"
          disabled={uploading}
        />
        <p className="text-xs text-gray-400 mt-1">Si vous avez des assets plus lourds, envoyez un lien Drive ou contactez-nous.</p>
      </div>

      <form onSubmit={handleAddLink} className="flex gap-2">
        <input name="link" className="flex-1 rounded-md px-3 py-2 bg-zinc-900/60 border border-white/10" placeholder="Ajouter un lien (YouTube, Drive...)" disabled={uploading} />
        <button className="px-3 py-2 bg-purple-600 rounded-md" disabled={uploading}>
          {uploading ? '...' : 'Ajouter'}
        </button>
      </form>

      {error && <div className="text-sm text-red-400">{error}</div>}

      <div className="space-y-2">
        {assets.map((a, idx) => (
          <div key={idx} className="flex items-center justify-between gap-2 p-2 bg-zinc-900/40 rounded-md">
            <div className="truncate">
              <div className="font-medium text-sm">{a.name}</div>
              <div className="text-xs text-gray-400">{a.type === 'file' ? `${Math.round((a.size || 0) / 1024)} KB` : a.url}</div>
            </div>
            <div>
              <button type="button" onClick={() => handleRemove(idx)} className="text-sm text-red-400">Supprimer</button>
            </div>
          </div>
        ))}
      </div>

      <div className="text-xs text-gray-400">Taille totale utilisée: {(totalBytes / 1024 / 1024).toFixed(2)} MB / {(maxTotalBytes / 1024 / 1024).toFixed(0)} MB</div>
    </div>
  )
}

export default AssetUpload
