'use client'

import { useState, useEffect } from 'react'
import { Button } from '@/components/ui/button'
import { Textarea } from './ui/textarea'
import { toast } from 'sonner'
import { Key, Loader2, Copy, Check } from 'lucide-react'

interface GameKey {
    id: string
    key: string
    platform: string
    isClaimed: number
    claimedBy: string | null
    claimedAt: string | null
}

export function GameKeysManager({ campaignId }: { campaignId: string }) {
    const [keys, setKeys] = useState<GameKey[]>([])
    const [loading, setLoading] = useState(true)
    const [newKeys, setNewKeys] = useState('')
    const [platform, setPlatform] = useState('PC')
    const [uploading, setUploading] = useState(false)

    useEffect(() => {
        loadKeys()
    }, [campaignId])

    const loadKeys = async () => {
        try {
            const res = await fetch(`/api/studio/campaigns/${campaignId}/keys`, {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                }
            })
            const data = await res.json()
            if (data.keys) {
                setKeys(data.keys)
            }
        } catch (error) {
            console.error('Error loading keys:', error)
        } finally {
            setLoading(false)
        }
    }

    const handleAddKeys = async () => {
        if (!newKeys.trim()) return

        setUploading(true)
        try {
            const keysList = newKeys.split('\n').map(k => k.trim()).filter(k => k)

            const res = await fetch(`/api/studio/campaigns/${campaignId}/keys`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                },
                body: JSON.stringify({ keys: keysList, platform })
            })

            const data = await res.json()
            if (res.ok) {
                toast.success(`${data.count} clés ajoutées !`)
                setNewKeys('')
                loadKeys()
            } else {
                toast.error(data.error || 'Erreur lors de l\'ajout')
            }
        } catch (error) {
            toast.error('Erreur serveur')
        } finally {
            setUploading(false)
        }
    }

    const claimedCount = keys.filter(k => k.isClaimed).length
    const availableCount = keys.length - claimedCount

    if (loading) return <div>Chargement des clés...</div>

    return (
        <div className="space-y-6">
            <div className="grid grid-cols-3 gap-4">
                <div className="bg-zinc-900/50 p-4 rounded-lg border border-white/5">
                    <div className="text-sm text-gray-400">Total</div>
                    <div className="text-2xl font-bold">{keys.length}</div>
                </div>
                <div className="bg-zinc-900/50 p-4 rounded-lg border border-white/5">
                    <div className="text-sm text-green-400">Disponibles</div>
                    <div className="text-2xl font-bold">{availableCount}</div>
                </div>
                <div className="bg-zinc-900/50 p-4 rounded-lg border border-white/5">
                    <div className="text-sm text-blue-400">Distribuées</div>
                    <div className="text-2xl font-bold">{claimedCount}</div>
                </div>
            </div>

            <div className="space-y-2">
                <label className="text-sm font-medium">Ajouter des clés (une par ligne)</label>
                <div className="flex gap-2 items-start">
                    <Textarea
                        value={newKeys}
                        onChange={(e) => setNewKeys(e.target.value)}
                        placeholder="XXXX-XXXX-XXXX&#10;YYYY-YYYY-YYYY"
                        className="font-mono text-sm flex-1"
                        rows={3}
                    />
                    <div className="flex flex-col gap-2">
                        <select
                            className="h-9 w-[120px] rounded-md border border-input bg-background px-3 py-1 text-sm shadow-sm transition-colors focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring"
                            value={platform}
                            onChange={(e) => setPlatform(e.target.value)}
                        >
                            <option value="PC">PC</option>
                            <option value="Xbox">Xbox</option>
                            <option value="PS4">PS4</option>
                            <option value="PS5">PS5</option>
                            <option value="Switch">Switch</option>
                        </select>
                        <Button
                            onClick={handleAddKeys}
                            disabled={uploading || !newKeys.trim()}
                            className="h-auto flex-1"
                        >
                            {uploading ? <Loader2 className="animate-spin" /> : 'Ajouter'}
                        </Button>
                    </div>
                </div>
            </div>

            <div className="border rounded-lg divide-y divide-white/10 max-h-60 overflow-y-auto">
                {keys.length === 0 ? (
                    <div className="p-4 text-center text-sm text-gray-500">Aucune clé configurée</div>
                ) : (
                    keys.map((key) => (
                        <div key={key.id} className="p-3 flex items-center justify-between text-sm">
                            <div className="font-mono text-gray-300 flex items-center gap-2">
                                <span className="text-xs px-1.5 py-0.5 rounded bg-white/10 text-gray-400">{key.platform || 'PC'}</span>
                                {key.key}
                            </div>
                            <div>
                                {key.isClaimed ? (
                                    <span className="px-2 py-1 rounded-full bg-blue-500/20 text-blue-400 text-xs">
                                        Distribuée
                                    </span>
                                ) : (
                                    <span className="px-2 py-1 rounded-full bg-green-500/20 text-green-400 text-xs">
                                        Disponible
                                    </span>
                                )}
                            </div>
                        </div>
                    ))
                )}
            </div>
        </div>
    )
}
