'use client'

import { useState, useRef } from 'react'
import { Button } from '@/components/ui/button'
import { ImageIcon, Loader2, Upload } from 'lucide-react'
import { toast } from 'sonner'

interface ImageUploadProps {
    onUploadComplete: (path: string) => void
    label?: string
    className?: string
}

export function ImageUpload({ onUploadComplete, label = "Upload Image", className }: ImageUploadProps) {
    const [loading, setLoading] = useState(false)
    const fileInputRef = useRef<HTMLInputElement>(null)

    const handleFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0]
        if (!file) return

        if (!file.type.startsWith('image/')) {
            toast.error('Veuillez sélectionner une image')
            return
        }

        if (file.size > 5 * 1024 * 1024) { // 5MB limit
            toast.error('L\'image ne doit pas dépasser 5MB')
            return
        }

        setLoading(true)
        const formData = new FormData()
        formData.append('file', file)
        // No campaignId needed, defaults to 'general'

        try {
            const response = await fetch('/api/upload', {
                method: 'POST',
                body: formData,
            })

            const data = await response.json()

            if (!response.ok) {
                throw new Error(data.error || 'Erreur lors de l\'upload')
            }

            onUploadComplete(data.path)
            toast.success('Image uploadée avec succès')
        } catch (error) {
            console.error('Upload error:', error)
            toast.error('Erreur lors de l\'upload')
        } finally {
            setLoading(false)
            // Reset input
            if (fileInputRef.current) {
                fileInputRef.current.value = ''
            }
        }
    }

    return (
        <div className={className}>
            <input
                type="file"
                ref={fileInputRef}
                onChange={handleFileChange}
                accept="image/*"
                className="hidden"
            />
            <Button
                type="button"
                variant="outline"
                className="w-full h-32 border-dashed border-2 border-white/20 hover:border-purple-500/50 hover:bg-purple-500/5 flex flex-col gap-2"
                onClick={() => fileInputRef.current?.click()}
                disabled={loading}
            >
                {loading ? (
                    <Loader2 className="w-8 h-8 animate-spin text-purple-500" />
                ) : (
                    <>
                        <ImageIcon className="w-8 h-8 text-gray-400" />
                        <span className="text-sm text-gray-400">{label}</span>
                    </>
                )}
            </Button>
        </div>
    )
}
