'use client'

import { useState } from 'react'
import { useTranslations } from 'next-intl'
import { Card } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Check, X, Eye } from 'lucide-react'
import { toast } from 'sonner'
import { formatCurrency } from '@/lib/utils'

interface Campaign {
    id: string
    title: string
    description: string
    budget: string
    pricePerView: string
    studio: {
        name: string | null
        company: string | null
    } | null
}

export function AdminCampaignList({ campaigns: initialCampaigns }: { campaigns: Campaign[] }) {
    const t = useTranslations('Ops.CampaignList')
    const [campaigns, setCampaigns] = useState(initialCampaigns)

    const handleAction = async (id: string, action: 'approve' | 'reject') => {
        try {
            const response = await fetch(`/api/admin/campaigns/${id}/${action}`, {
                method: 'POST'
            })

            if (response.ok) {
                toast.success(action === 'approve' ? t('success.approved') : t('success.rejected'))
                setCampaigns(campaigns.filter(c => c.id !== id))
            } else {
                toast.error(t('errors.action'))
            }
        } catch (error) {
            console.error(error)
            toast.error(t('errors.server'))
        }
    }

    if (campaigns.length === 0) {
        return (
            <div className="text-center text-gray-400 py-8">
                {t('empty')}
            </div>
        )
    }

    return (
        <div className="space-y-4">
            {campaigns.map((campaign) => (
                <div key={campaign.id} className="flex items-center justify-between p-4 bg-white/5 rounded-xl border border-white/10">
                    <div>
                        <h3 className="font-bold text-lg">{campaign.title}</h3>
                        <p className="text-sm text-gray-400">
                            {t('by', { studio: campaign.studio?.company || campaign.studio?.name || t('unknown_studio') })}
                        </p>
                        <div className="flex gap-4 mt-2 text-sm">
                            <span className="text-purple-400">{t('budget', { amount: formatCurrency(campaign.budget) })}</span>
                            <span className="text-blue-400">{t('ppv', { amount: formatCurrency(campaign.pricePerView) })}</span>
                        </div>
                    </div>
                    <div className="flex gap-2">
                        <Button
                            size="sm"
                            className="bg-green-500/20 text-green-400 hover:bg-green-500/30"
                            onClick={() => handleAction(campaign.id, 'approve')}
                        >
                            <Check className="w-4 h-4" />
                        </Button>
                        <Button
                            size="sm"
                            className="bg-red-500/20 text-red-400 hover:bg-red-500/30"
                            onClick={() => handleAction(campaign.id, 'reject')}
                        >
                            <X className="w-4 h-4" />
                        </Button>
                    </div>
                </div>
            ))}
        </div>
    )
}
