'use client'

import { useState } from 'react'
import { Button } from '@/components/ui/button'
import { toast } from 'sonner'
import { approveUser, rejectUser } from '@/app/actions/admin-moderation'
import { suspendCreator, deleteCreator } from '@/app/actions/admin-creators'
import { CheckCircle, XCircle, Ban, Trash2 } from 'lucide-react'
import { useRouter } from 'next/navigation'

interface AdminUserActionsProps {
    userId: string
    isApproved: boolean
    userName: string
    role: 'creator' | 'studio'
}

export function AdminUserActions({ userId, isApproved, userName, role }: AdminUserActionsProps) {
    const [loading, setLoading] = useState(false)
    const router = useRouter()

    const handleApprove = async () => {
        setLoading(true)
        try {
            const result = await approveUser(userId)
            if (result.success) {
                toast.success(`Utilisateur ${userName} approuvé !`)
                router.refresh()
            } else {
                toast.error('Erreur lors de l\'approbation')
            }
        } catch (error) {
            toast.error('Erreur serveur')
        } finally {
            setLoading(false)
        }
    }

    const handleReject = async () => {
        if (!confirm(`Voulez-vous vraiment rejeter (supprimer) ${userName} ?`)) return
        setLoading(true)
        try {
            const result = await rejectUser(userId)
            if (result.success) {
                toast.success(`Utilisateur ${userName} rejeté !`)
                router.push(role === 'creator' ? '/ops/creators' : '/ops/studios')
            } else {
                toast.error('Erreur lors du rejet')
            }
        } catch (error) {
            toast.error('Erreur serveur')
        } finally {
            setLoading(false)
        }
    }

    const handleSuspend = async () => {
        if (!confirm(`Voulez-vous vraiment suspendre ${userName} ?`)) return
        setLoading(true)
        try {
            // suspendCreator works for any user if we just update isApproved=0
            // But the action is named suspendCreator. Let's assume it works for studios too if it just updates users table.
            // Checking admin-creators.ts: yes, it updates 'users' table.
            const result = await suspendCreator(userId)
            if (result.success) {
                toast.success(`Utilisateur ${userName} suspendu !`)
                router.refresh()
            } else {
                toast.error('Erreur lors de la suspension')
            }
        } catch (error) {
            toast.error('Erreur serveur')
        } finally {
            setLoading(false)
        }
    }

    const handleDelete = async () => {
        if (!confirm(`Voulez-vous vraiment SUPPRIMER DÉFINITIVEMENT ${userName} ?`)) return
        setLoading(true)
        try {
            const result = await deleteCreator(userId) // Also works for any user
            if (result.success) {
                toast.success(`Utilisateur ${userName} supprimé !`)
                router.push(role === 'creator' ? '/ops/creators' : '/ops/studios')
            } else {
                toast.error('Erreur lors de la suppression')
            }
        } catch (error) {
            toast.error('Erreur serveur')
        } finally {
            setLoading(false)
        }
    }

    if (!isApproved) {
        return (
            <div className="flex gap-2">
                <Button
                    onClick={handleApprove}
                    disabled={loading}
                    className="bg-green-600 hover:bg-green-700 text-white"
                >
                    <CheckCircle className="w-4 h-4 mr-2" />
                    Approuver
                </Button>
                <Button
                    onClick={handleReject}
                    disabled={loading}
                    variant="destructive"
                >
                    <XCircle className="w-4 h-4 mr-2" />
                    Rejeter
                </Button>
            </div>
        )
    }

    return (
        <div className="flex gap-2">
            <Button
                onClick={handleSuspend}
                disabled={loading}
                variant="outline"
                className="border-orange-500/30 text-orange-400 hover:bg-orange-500/10"
            >
                <Ban className="w-4 h-4 mr-2" />
                Suspendre
            </Button>
            <Button
                onClick={handleDelete}
                disabled={loading}
                variant="outline"
                className="border-red-500/30 text-red-400 hover:bg-red-500/10"
            >
                <Trash2 className="w-4 h-4 mr-2" />
                Supprimer
            </Button>
        </div>
    )
}
