'use client'

import { useState } from 'react'
import { Button } from '@/components/ui/button'
import { Badge } from '@/components/ui/badge'
import { ExternalLink, Check, X, Loader2, Instagram, Youtube, Globe, RefreshCw, Smartphone, Image as ImageIcon } from 'lucide-react'
import { verifySocial, rejectSocial } from '@/app/actions/admin-creators'
import { refreshCreatorStats } from '@/app/actions/social-stats'
import { toast } from 'sonner'
import { Dialog, DialogContent, DialogTrigger, DialogTitle, DialogDescription } from '@/components/ui/dialog'

interface Social {
    id: string
    creatorId: string
    platform: string
    handle: string
    profileUrl: string | null
    verificationStatus: 'pending' | 'approved' | 'rejected'
    followers: number | null
    avgViews: number | null
    screenshotUrl: string | null
    verificationMethod: string | null
}

export function CreatorSocialsList({ socials }: { socials: Social[] }) {
    const [processing, setProcessing] = useState<string | null>(null)
    const [refreshing, setRefreshing] = useState(false)

    const handleVerify = async (id: string) => {
        setProcessing(id)
        try {
            const res = await verifySocial(id)
            if (res.success) {
                toast.success('Réseau social vérifié')
            } else {
                toast.error('Erreur lors de la vérification')
            }
        } catch (error) {
            toast.error('Erreur serveur')
        } finally {
            setProcessing(null)
        }
    }

    const handleReject = async (id: string) => {
        setProcessing(id)
        try {
            const res = await rejectSocial(id, 'Non conforme')
            if (res.success) {
                toast.success('Réseau social rejeté')
            } else {
                toast.error('Erreur lors du rejet')
            }
        } catch (error) {
            toast.error('Erreur serveur')
        } finally {
            setProcessing(null)
        }
    }

    const handleRefreshStats = async () => {
        if (socials.length === 0) return
        setRefreshing(true)
        try {
            const res = await refreshCreatorStats(socials[0].creatorId)
            if (res.success) {
                toast.success(`${res.updatedCount} réseaux mis à jour`)
            } else {
                toast.error('Erreur lors de la mise à jour')
            }
        } catch (error) {
            toast.error('Erreur serveur')
        } finally {
            setRefreshing(false)
        }
    }

    const getPlatformIcon = (platform: string) => {
        switch (platform) {
            case 'instagram': return <Instagram className="w-5 h-5 text-pink-500" />
            case 'youtube': return <Youtube className="w-5 h-5 text-red-500" />
            case 'tiktok': return <svg className="w-5 h-5" viewBox="0 0 24 24" fill="currentColor"><path d="M19.59 6.69a4.83 4.83 0 01-3.77-4.25V2h-3.45v13.67a2.89 2.89 0 01-5.2 1.74 2.89 2.89 0 012.31-4.64 2.93 2.93 0 01.88.13V9.4a6.84 6.84 0 00-1-.05A6.33 6.33 0 005 20.1a6.34 6.34 0 0010.86-4.43v-7a8.16 8.16 0 004.77 1.52v-3.4a4.85 4.85 0 01-1-.1z" /></svg>
            default: return <Globe className="w-5 h-5 text-gray-400" />
        }
    }

    if (socials.length === 0) {
        return <div className="text-center py-8 text-gray-500">Aucun réseau social connecté</div>
    }

    return (
        <div className="space-y-4">
            <div className="flex justify-end mb-4">
                <Button
                    variant="outline"
                    size="sm"
                    onClick={handleRefreshStats}
                    disabled={refreshing}
                    className="border-white/10 hover:bg-white/5"
                >
                    {refreshing ? <Loader2 className="w-4 h-4 animate-spin mr-2" /> : <RefreshCw className="w-4 h-4 mr-2" />}
                    Actualiser les stats
                </Button>
            </div>

            {socials.map((social) => (
                <div key={social.id} className="flex flex-col md:flex-row items-start md:items-center justify-between p-4 bg-white/5 rounded-lg border border-white/10 gap-4">
                    <div className="flex items-center gap-4 flex-1">
                        <div className="p-3 bg-white/5 rounded-lg">
                            {getPlatformIcon(social.platform)}
                        </div>
                        <div>
                            <div className="flex items-center gap-2">
                                <span className="font-medium capitalize">{social.platform}</span>
                                {social.verificationStatus === 'approved' && (
                                    <Badge className="bg-green-500/20 text-green-400 border-0 text-xs">Vérifié</Badge>
                                )}
                                {social.verificationStatus === 'pending' && (
                                    <Badge className="bg-yellow-500/20 text-yellow-400 border-0 text-xs">En attente</Badge>
                                )}
                                {social.verificationStatus === 'rejected' && (
                                    <Badge className="bg-red-500/20 text-red-400 border-0 text-xs">Rejeté</Badge>
                                )}
                            </div>
                            <div className="text-sm text-gray-400 mt-1">{social.handle}</div>
                            <div className="flex items-center gap-3 mt-1">
                                {social.profileUrl && (
                                    <a href={social.profileUrl} target="_blank" rel="noopener noreferrer" className="text-xs text-blue-400 hover:underline flex items-center gap-1">
                                        Voir le profil <ExternalLink className="w-3 h-3" />
                                    </a>
                                )}
                                {social.verificationMethod === 'screenshot' && (
                                    <span className="text-xs text-gray-500 flex items-center gap-1 bg-white/5 px-1.5 py-0.5 rounded">
                                        <ImageIcon className="w-3 h-3" /> Screenshot
                                    </span>
                                )}
                                {social.verificationMethod === 'oauth' && (
                                    <span className="text-xs text-gray-500 flex items-center gap-1 bg-white/5 px-1.5 py-0.5 rounded">
                                        <Smartphone className="w-3 h-3" /> OAuth
                                    </span>
                                )}
                            </div>
                        </div>
                    </div>

                    <div className="flex items-center gap-4">
                        <div className="text-right hidden md:block">
                            <div className="text-sm font-medium">{social.followers?.toLocaleString() || 0} abonnés</div>
                            <div className="text-xs text-gray-500">{social.avgViews?.toLocaleString() || 0} vues moy.</div>
                        </div>

                        {social.screenshotUrl && (
                            <Dialog>
                                <DialogTrigger asChild>
                                    <button className="relative group">
                                        <img src={social.screenshotUrl} alt="Proof" className="h-12 w-12 rounded object-cover border border-white/10 group-hover:opacity-80 transition-opacity" />
                                        <div className="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity bg-black/50 rounded">
                                            <ExternalLink className="w-4 h-4 text-white" />
                                        </div>
                                    </button>
                                </DialogTrigger>
                                <DialogContent className="max-w-3xl bg-zinc-900 border-white/10 p-0 overflow-hidden">
                                    <DialogTitle className="sr-only">Preuve de vérification</DialogTitle>
                                    <DialogDescription className="sr-only">
                                        Capture d'écran de la preuve de vérification du compte social.
                                    </DialogDescription>
                                    <img src={social.screenshotUrl} alt="Proof Full" className="w-full h-auto" />
                                </DialogContent>
                            </Dialog>
                        )}

                        <div className="flex gap-2">
                            {social.verificationStatus === 'pending' && (
                                <>
                                    <Button
                                        size="sm"
                                        className="bg-green-600 hover:bg-green-700 h-8 w-8 p-0"
                                        onClick={() => handleVerify(social.id)}
                                        disabled={processing === social.id}
                                        title="Approuver"
                                    >
                                        {processing === social.id ? <Loader2 className="w-4 h-4 animate-spin" /> : <Check className="w-4 h-4" />}
                                    </Button>
                                    <Button
                                        size="sm"
                                        variant="destructive"
                                        className="h-8 w-8 p-0"
                                        onClick={() => handleReject(social.id)}
                                        disabled={processing === social.id}
                                        title="Rejeter"
                                    >
                                        {processing === social.id ? <Loader2 className="w-4 h-4 animate-spin" /> : <X className="w-4 h-4" />}
                                    </Button>
                                </>
                            )}
                            {social.verificationStatus === 'rejected' && (
                                <Button
                                    size="sm"
                                    variant="outline"
                                    className="h-8 w-8 p-0 border-green-500/30 text-green-400 hover:bg-green-500/10"
                                    onClick={() => handleVerify(social.id)}
                                    disabled={processing === social.id}
                                    title="Re-vérifier"
                                >
                                    <Check className="w-4 h-4" />
                                </Button>
                            )}
                        </div>
                    </div>
                </div>
            ))}
        </div>
    )
}
