'use client'

import { useState } from 'react'
import { Button } from '@/components/ui/button'
import { Check, X, Loader2 } from 'lucide-react'
import { toast } from 'sonner'
import { approveUser, rejectUser, approveCampaign, rejectCampaign } from '@/app/actions/admin-moderation'

interface ModerationActionsProps {
    id: string
    type: 'creator' | 'campaign'
}

export function ModerationActions({ id, type }: ModerationActionsProps) {
    const [loading, setLoading] = useState(false)

    const handleApprove = async () => {
        setLoading(true)
        try {
            let res
            if (type === 'creator') {
                res = await approveUser(id)
            } else {
                res = await approveCampaign(id)
            }

            if (res.error) {
                toast.error(res.error)
            } else {
                toast.success(`${type === 'creator' ? 'Utilisateur' : 'Campagne'} approuvé(e) !`)
            }
        } catch (error) {
            toast.error('Erreur lors de l\'approbation')
        } finally {
            setLoading(false)
        }
    }

    const handleReject = async () => {
        if (!confirm(`Êtes-vous sûr de vouloir rejeter ${type === 'creator' ? 'cet utilisateur' : 'cette campagne'} ?`)) return

        setLoading(true)
        try {
            let res
            if (type === 'creator') {
                res = await rejectUser(id)
            } else {
                res = await rejectCampaign(id)
            }

            if (res.error) {
                toast.error(res.error)
            } else {
                toast.success(`${type === 'creator' ? 'Utilisateur' : 'Campagne'} rejeté(e)`)
            }
        } catch (error) {
            toast.error('Erreur lors du rejet')
        } finally {
            setLoading(false)
        }
    }

    return (
        <div className="flex gap-3">
            <Button
                onClick={handleApprove}
                disabled={loading}
                className="bg-green-600 hover:bg-green-700 text-white"
            >
                {loading ? <Loader2 className="w-4 h-4 animate-spin" /> : <Check className="w-4 h-4 mr-2" />}
                Approuver
            </Button>
            <Button
                onClick={handleReject}
                disabled={loading}
                variant="outline"
                className="border-red-500/30 text-red-400 hover:bg-red-500/10 hover:text-red-300"
            >
                {loading ? <Loader2 className="w-4 h-4 animate-spin" /> : <X className="w-4 h-4 mr-2" />}
                Rejeter
            </Button>
        </div>
    )
}
