'use client'

import { useState } from 'react'
import { useTranslations } from 'next-intl'
import { Card } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { updateAdminSettings } from '@/app/actions/admin-settings'
import { toast } from 'sonner'
import {
    Settings as SettingsIcon,
    Lock,
    DollarSign,
    Users,
    Save,
    Webhook,
    CreditCard,
    Euro,
    DollarSignIcon
} from 'lucide-react'

interface SettingsFormProps {
    initialSettings: {
        platformFee: number
        minPayout: number
        autoApproveCreators: boolean
        requireStudioApproval: boolean
        emailNotifications: boolean
        weeklyReports: boolean
        maintenanceMode: boolean
        autoMonthlyInvoicing: boolean
        discordWebhookUrl?: string | null
        // Bank EUR
        bankEurBeneficiary?: string | null
        bankEurIban?: string | null
        bankEurBic?: string | null
        bankEurBankName?: string | null
        bankEurBankAddress?: string | null
        // Bank USD
        bankUsdBeneficiary?: string | null
        bankUsdAccountNumber?: string | null
        bankUsdRoutingNumber?: string | null
        bankUsdBic?: string | null
        bankUsdAccountType?: string | null
        bankUsdBankName?: string | null
        bankUsdBankAddress?: string | null
    }
}

export function SettingsForm({ initialSettings }: SettingsFormProps) {
    const t = useTranslations('Ops.Settings')
    const [saving, setSaving] = useState(false)
    const [settings, setSettings] = useState({
        ...initialSettings,
        platformFee: 20
    })

    const handleSave = async () => {
        setSaving(true)
        try {
            const result = await updateAdminSettings(settings)
            if (result.success) {
                toast.success(t('toasts.save_success'))
            } else {
                toast.error(t('toasts.save_error'))
            }
        } catch (error) {
            toast.error(t('toasts.save_error'))
        } finally {
            setSaving(false)
        }
    }

    return (
        <div className="space-y-6 max-w-4xl pt-6">
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-bold mb-2">{t('title')}</h1>
                    <p className="text-gray-400">{t('subtitle')}</p>
                </div>
                <Button
                    onClick={handleSave}
                    disabled={saving}
                    className="bg-purple-600 hover:bg-purple-700"
                >
                    <Save className="w-4 h-4 mr-2" />
                    {saving ? t('actions.saving') : t('actions.save')}
                </Button>
            </div>

            {/* Bank Details EUR */}
            <Card className="p-6 bg-white/[0.03] border-white/10">
                <h3 className="text-xl font-semibold mb-4 flex items-center gap-2">
                    <Euro className="w-5 h-5 text-blue-400" />
                    Coordonnées Bancaires EUR (Wise)
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <Label className="text-gray-400">Bénéficiaire</Label>
                        <Input
                            value={settings.bankEurBeneficiary || ''}
                            onChange={(e) => setSettings({ ...settings, bankEurBeneficiary: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                    <div>
                        <Label className="text-gray-400">IBAN</Label>
                        <Input
                            value={settings.bankEurIban || ''}
                            onChange={(e) => setSettings({ ...settings, bankEurIban: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                    <div>
                        <Label className="text-gray-400">BIC</Label>
                        <Input
                            value={settings.bankEurBic || ''}
                            onChange={(e) => setSettings({ ...settings, bankEurBic: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                    <div>
                        <Label className="text-gray-400">Nom Banque</Label>
                        <Input
                            value={settings.bankEurBankName || ''}
                            onChange={(e) => setSettings({ ...settings, bankEurBankName: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                    <div className="md:col-span-2">
                        <Label className="text-gray-400">Adresse Banque</Label>
                        <Input
                            value={settings.bankEurBankAddress || ''}
                            onChange={(e) => setSettings({ ...settings, bankEurBankAddress: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                </div>
            </Card>

            {/* Bank Details USD */}
            <Card className="p-6 bg-white/[0.03] border-white/10">
                <h3 className="text-xl font-semibold mb-4 flex items-center gap-2">
                    <DollarSignIcon className="w-5 h-5 text-green-400" />
                    Coordonnées Bancaires USD (Wise)
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <Label className="text-gray-400">Bénéficiaire</Label>
                        <Input
                            value={settings.bankUsdBeneficiary || ''}
                            onChange={(e) => setSettings({ ...settings, bankUsdBeneficiary: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                    <div>
                        <Label className="text-gray-400">Account Number</Label>
                        <Input
                            value={settings.bankUsdAccountNumber || ''}
                            onChange={(e) => setSettings({ ...settings, bankUsdAccountNumber: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                    <div>
                        <Label className="text-gray-400">Routing Number (ACH)</Label>
                        <Input
                            value={settings.bankUsdRoutingNumber || ''}
                            onChange={(e) => setSettings({ ...settings, bankUsdRoutingNumber: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                    <div>
                        <Label className="text-gray-400">BIC / SWIFT</Label>
                        <Input
                            value={settings.bankUsdBic || ''}
                            onChange={(e) => setSettings({ ...settings, bankUsdBic: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                    <div>
                        <Label className="text-gray-400">Type de compte</Label>
                        <Input
                            value={settings.bankUsdAccountType || ''}
                            onChange={(e) => setSettings({ ...settings, bankUsdAccountType: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                    <div>
                        <Label className="text-gray-400">Nom Banque</Label>
                        <Input
                            value={settings.bankUsdBankName || ''}
                            onChange={(e) => setSettings({ ...settings, bankUsdBankName: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                    <div className="md:col-span-2">
                        <Label className="text-gray-400">Adresse Banque</Label>
                        <Input
                            value={settings.bankUsdBankAddress || ''}
                            onChange={(e) => setSettings({ ...settings, bankUsdBankAddress: e.target.value })}
                            className="mt-1 bg-white/[0.03] border-white/10"
                        />
                    </div>
                </div>
            </Card>

            <Card className="p-6 bg-white/[0.03] border-white/10">
                <h3 className="text-xl font-semibold mb-4 flex items-center gap-2">
                    <DollarSign className="w-5 h-5 text-green-400" />
                    {t('sections.financial.title')}
                </h3>
                <div className="space-y-4">
                    <div>
                        <label className="block text-sm text-gray-400 mb-2">
                            {t('sections.financial.platform_fee')} (%)
                        </label>
                        <input
                            type="number"
                            value={20}
                            disabled
                            className="w-full px-4 py-2 bg-white/[0.03] border border-white/10 rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-500 opacity-50 cursor-not-allowed"
                        />
                        <p className="text-xs text-gray-500 mt-1">
                            La marge est fixée à 20% et ne peut pas être modifiée.
                        </p>
                    </div>
                </div>
            </Card>

            <Card className="p-6 bg-white/[0.03] border-white/10">
                <h3 className="text-xl font-semibold mb-4 flex items-center gap-2">
                    <Users className="w-5 h-5 text-purple-400" />
                    {t('sections.users.title')}
                </h3>
                <div className="space-y-4">
                    <div className="flex items-center justify-between p-4 bg-white/[0.03] rounded-lg">
                        <div>
                            <div className="font-medium">{t('sections.users.auto_approve_creators')}</div>
                            <div className="text-sm text-gray-400">{t('sections.users.auto_approve_creators_desc')}</div>
                        </div>
                        <button
                            onClick={() => setSettings({ ...settings, autoApproveCreators: !settings.autoApproveCreators })}
                            className={`w-12 h-6 rounded-full transition-colors ${settings.autoApproveCreators ? 'bg-green-500' : 'bg-gray-600'}`}
                        >
                            <div className={`w-5 h-5 bg-white rounded-full transition-transform ${settings.autoApproveCreators ? 'translate-x-6' : 'translate-x-1'}`} />
                        </button>
                    </div>

                    <div className="flex items-center justify-between p-4 bg-white/[0.03] rounded-lg">
                        <div>
                            <div className="font-medium">{t('sections.users.manual_approve_studios')}</div>
                            <div className="text-sm text-gray-400">{t('sections.users.manual_approve_studios_desc')}</div>
                        </div>
                        <button
                            onClick={() => setSettings({ ...settings, requireStudioApproval: !settings.requireStudioApproval })}
                            className={`w-12 h-6 rounded-full transition-colors ${settings.requireStudioApproval ? 'bg-green-500' : 'bg-gray-600'}`}
                        >
                            <div className={`w-5 h-5 bg-white rounded-full transition-transform ${settings.requireStudioApproval ? 'translate-x-6' : 'translate-x-1'}`} />
                        </button>
                    </div>
                </div>
            </Card>

            <Card className="p-6 bg-white/[0.03] border-white/10">
                <h3 className="text-xl font-semibold mb-4 flex items-center gap-2">
                    <Webhook className="w-5 h-5 text-blue-400" />
                    Notifications Discord
                </h3>
                <div className="space-y-4">
                    <div>
                        <label className="block text-sm text-gray-400 mb-2">Webhook URL</label>
                        <input
                            type="text"
                            value={settings.discordWebhookUrl || ''}
                            onChange={(e) => setSettings({ ...settings, discordWebhookUrl: e.target.value })}
                            placeholder="https://discord.com/api/webhooks/..."
                            className="w-full px-4 py-2 bg-white/[0.03] border border-white/10 rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-500"
                        />
                    </div>
                </div>
            </Card>

            <Card className="p-6 bg-white/[0.03] border-white/10 border-red-500/30">
                <h3 className="text-xl font-semibold mb-4 flex items-center gap-2">
                    <Lock className="w-5 h-5 text-red-400" />
                    {t('sections.danger.title')}
                </h3>
                <div className="space-y-4">
                    <div className="flex items-center justify-between p-4 bg-red-500/10 rounded-lg">
                        <div>
                            <div className="font-medium text-red-400">{t('sections.danger.maintenance')}</div>
                            <div className="text-sm text-gray-400">{t('sections.danger.maintenance_desc')}</div>
                        </div>
                        <button
                            onClick={() => setSettings({ ...settings, maintenanceMode: !settings.maintenanceMode })}
                            className={`w-12 h-6 rounded-full transition-colors ${settings.maintenanceMode ? 'bg-red-500' : 'bg-gray-600'}`}
                        >
                            <div className={`w-5 h-5 bg-white rounded-full transition-transform ${settings.maintenanceMode ? 'translate-x-6' : 'translate-x-1'}`} />
                        </button>
                    </div>
                </div>
            </Card>
        </div>
    )
}

