'use client'

import { useState, useEffect } from 'react'
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, AreaChart, Area } from 'recharts'
import { Card } from '@/components/ui/card'
import { Tabs, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Loader2 } from 'lucide-react'
import { format } from 'date-fns'
import { fr } from 'date-fns/locale'

interface StatsHistoryChartProps {
    submissionId: string
}

export function StatsHistoryChart({ submissionId }: StatsHistoryChartProps) {
    const [data, setData] = useState<any[]>([])
    const [loading, setLoading] = useState(true)
    const [metric, setMetric] = useState('views')

    useEffect(() => {
        const fetchData = async () => {
            try {
                const response = await fetch(`/api/creator/submissions/${submissionId}/history`)
                if (response.ok) {
                    const json = await response.json()
                    setData(json.history)
                }
            } catch (error) {
                console.error('Error fetching history:', error)
            } finally {
                setLoading(false)
            }
        }
        fetchData()
    }, [submissionId])

    if (loading) {
        return <div className="flex justify-center p-8"><Loader2 className="w-6 h-6 animate-spin text-purple-500" /></div>
    }

    if (data.length === 0) {
        return <div className="text-center p-8 text-gray-500">Pas encore d'historique disponible</div>
    }

    const formatDate = (dateStr: string) => {
        return format(new Date(dateStr), 'dd/MM HH:h', { locale: fr })
    }

    const getMetricConfig = () => {
        switch (metric) {
            case 'views':
                return { color: '#8b5cf6', name: 'Vues', key: 'views' }
            case 'likes':
                return { color: '#ec4899', name: 'Likes', key: 'likes' }
            case 'engagement':
                return { color: '#3b82f6', name: 'Engagement (%)', key: 'engagementRate' }
            default:
                return { color: '#8b5cf6', name: 'Vues', key: 'views' }
        }
    }

    const config = getMetricConfig()

    return (
        <Card className="p-4 bg-zinc-900/50 border-white/5">
            <div className="flex items-center justify-between mb-6">
                <h3 className="font-semibold text-white">Évolution des performances</h3>
                <Tabs value={metric} onValueChange={setMetric}>
                    <TabsList className="bg-zinc-800">
                        <TabsTrigger value="views">Vues</TabsTrigger>
                        <TabsTrigger value="likes">Likes</TabsTrigger>
                        <TabsTrigger value="engagement">Engagement</TabsTrigger>
                    </TabsList>
                </Tabs>
            </div>

            <div className="h-[300px] w-full">
                <ResponsiveContainer width="100%" height="100%">
                    <AreaChart data={data}>
                        <defs>
                            <linearGradient id={`color${metric}`} x1="0" y1="0" x2="0" y2="1">
                                <stop offset="5%" stopColor={config.color} stopOpacity={0.3} />
                                <stop offset="95%" stopColor={config.color} stopOpacity={0} />
                            </linearGradient>
                        </defs>
                        <CartesianGrid strokeDasharray="3 3" stroke="#333" vertical={false} />
                        <XAxis
                            dataKey="recordedAt"
                            tickFormatter={formatDate}
                            stroke="#666"
                            fontSize={12}
                            tickMargin={10}
                        />
                        <YAxis
                            stroke="#666"
                            fontSize={12}
                            tickFormatter={(value) => value.toLocaleString()}
                        />
                        <Tooltip
                            contentStyle={{ backgroundColor: '#18181b', borderColor: '#333', borderRadius: '8px' }}
                            itemStyle={{ color: '#fff' }}
                            labelStyle={{ color: '#999', marginBottom: '4px' }}
                            labelFormatter={(label) => format(new Date(label), 'dd MMMM HH:mm', { locale: fr })}
                            formatter={(value: any) => [value.toLocaleString(), config.name]}
                        />
                        <Area
                            type="monotone"
                            dataKey={config.key}
                            stroke={config.color}
                            fillOpacity={1}
                            fill={`url(#color${metric})`}
                            strokeWidth={2}
                        />
                    </AreaChart>
                </ResponsiveContainer>
            </div>
        </Card>
    )
}
