'use client'

import { useState } from 'react'
import { Link } from '@/i18n/routing'
import { usePathname } from 'next/navigation'
import Image from 'next/image'
import { cn } from '@/lib/utils'
import {
    LayoutDashboard, Upload, DollarSign, User, Settings, BarChart3,
    Megaphone, Users, CreditCard, HelpCircle,
    Building2, FileText, ShieldAlert,
    LogOut, X, LucideIcon
} from 'lucide-react'
import { signOut } from 'next-auth/react'

// Icon mapping to avoid passing functions from server to client
const iconMap: Record<string, LucideIcon> = {
    LayoutDashboard,
    Upload,
    DollarSign,
    User,
    Settings,
    BarChart3,
    Megaphone,
    Users,
    CreditCard,
    HelpCircle,
    Building2,
    FileText,
    ShieldAlert
}

interface SidebarItem {
    iconName: string
    label: string
    href: string
}

interface DashboardSidebarProps {
    items: SidebarItem[]
    role: 'creator' | 'studio' | 'admin'
    user?: any
    isOpen?: boolean
    onClose?: () => void
}

import { useTranslations } from 'next-intl'
import { useAuth } from '@/lib/auth-store'

// ... imports

export function DashboardSidebar({ items, role, user, isOpen = false, onClose }: DashboardSidebarProps) {
    const t = useTranslations('Dashboard.Sidebar')
    const pathname = usePathname()
    // Internal state removed in favor of props
    const storedUser = useAuth((state) => state.user)
    const displayUser = storedUser || user

    const handleLogout = () => {
        signOut({ callbackUrl: 'https://swiplay.com' })
    }

    const filteredItems = items

    return (
        <>
            {/* Mobile Overlay */}
            {isOpen && (
                <div
                    className="fixed inset-0 bg-black/80 backdrop-blur-sm z-40 md:hidden"
                    onClick={onClose}
                />
            )}

            {/* Sidebar */}
            <aside className={cn(
                "fixed left-0 top-0 bottom-0 w-64 bg-black/90 backdrop-blur-xl border-r border-white/10 z-50 flex flex-col transition-transform duration-300",
                "md:translate-x-0",
                isOpen ? "translate-x-0" : "-translate-x-full"
            )}>
                {/* Header */}
                <div className="p-6 flex items-center justify-between">
                    <Link href={role === 'creator' ? '/creator/dashboard' : role === 'studio' ? '/studio/dashboard' : '/ops'} className="flex items-center gap-2">
                        <div className="w-8 h-8 relative flex items-center justify-center">
                            <Image
                                src="/swiplay-logo.png"
                                alt="Swiplay Logo"
                                width={32}
                                height={32}
                                className="object-contain"
                            />
                        </div>
                        <span className="text-xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-white to-gray-400">
                            Swiplay
                        </span>
                    </Link>
                    <button onClick={onClose} className="md:hidden text-gray-400 hover:text-white">
                        <X className="w-6 h-6" />
                    </button>
                </div>

                {/* Navigation */}
                <nav className="flex-1 px-4 space-y-2 overflow-y-auto">
                    {filteredItems.map((item) => {
                        const Icon = iconMap[item.iconName] || HelpCircle
                        const isActive = pathname === item.href

                        return (
                            <Link
                                key={item.href}
                                href={item.href}
                                onClick={onClose}
                                className={cn(
                                    "flex items-center gap-3 px-4 py-3 rounded-xl transition-all duration-300 group",
                                    isActive
                                        ? "bg-white/10 text-white shadow-lg shadow-purple-500/10"
                                        : "text-gray-400 hover:text-white hover:bg-white/5"
                                )}
                            >
                                <Icon className={cn(
                                    "w-5 h-5 transition-colors",
                                    isActive ? "text-purple-400" : "text-gray-500 group-hover:text-purple-400"
                                )} />
                                <span className="font-medium">{item.label}</span>
                                {isActive && (
                                    <div className="ml-auto w-1.5 h-1.5 rounded-full bg-purple-400 shadow-[0_0_8px_rgba(192,132,252,0.5)]" />
                                )}
                            </Link>
                        )
                    })}
                </nav>

                {/* User Profile / Footer */}
                <div className="p-4 border-t border-white/5 space-y-2">
                    {/* Only show profile info if onboarding is complete */}
                    <div className="glass p-4 rounded-xl flex items-center gap-3">
                        {displayUser?.image ? (
                            <Image
                                src={displayUser.image}
                                alt={displayUser.name || 'User'}
                                width={40}
                                height={40}
                                className="w-10 h-10 rounded-full object-cover"
                            />
                        ) : (
                            <div className={cn(
                                "w-10 h-10 rounded-full flex items-center justify-center font-bold text-white",
                                role === 'studio' ? "bg-blue-600" : role === 'admin' ? "bg-red-600" : "bg-purple-600"
                            )}>
                                {displayUser?.name?.[0]?.toUpperCase() || role[0].toUpperCase()}
                            </div>
                        )}
                        <div className="flex-1 min-w-0">
                            <p className="text-sm font-medium text-white truncate capitalize">
                                {role === 'studio' ? (displayUser?.company || displayUser?.name) : (displayUser?.name || role)}
                            </p>
                        </div>
                    </div>

                    <button
                        onClick={handleLogout}
                        className="w-full flex items-center gap-3 px-4 py-3 rounded-xl text-gray-400 hover:text-white hover:bg-red-500/10 transition-all duration-300"
                    >
                        <LogOut className="w-5 h-5" />
                        <span className="font-medium">{t('logout')}</span>
                    </button>
                </div>
            </aside>
        </>
    )
}
