'use client'

import { useState } from 'react'
import { useTranslations, useLocale } from 'next-intl'
import { X, Mail, ArrowRight, Loader2 } from 'lucide-react'
import Image from 'next/image'
import { Button } from '@/components/ui/button'
import { cn } from '@/lib/utils'

interface AuthModalProps {
    isOpen: boolean
    onClose: () => void
    defaultRole?: 'studio' | 'creator'
}

export function AuthModal({ isOpen, onClose, defaultRole = 'studio' }: AuthModalProps) {
    const t = useTranslations('Auth')
    const locale = useLocale()
    const [role, setRole] = useState<'studio' | 'creator'>(defaultRole)
    const [email, setEmail] = useState('')
    const [loading, setLoading] = useState(false)
    const [message, setMessage] = useState('')
    const [acceptedTerms, setAcceptedTerms] = useState(false)
    const [acceptedPrivacy, setAcceptedPrivacy] = useState(false)

    if (!isOpen) return null

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault()
        if (!email) {
            setMessage(t('email_required'))
            return
        }

        setLoading(true)
        setMessage('')

        try {
            const response = await fetch('/api/auth/send-code', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ email, role })
            })

            const data = await response.json()

            if (!response.ok) {
                setMessage(data.error || t('error_send'))
                setLoading(false)
                return
            }

            window.location.href = `/verify?email=${encodeURIComponent(email)}&role=${role}&new=${data.isNewAccount || false}`
        } catch (error) {
            setMessage(t('error_network'))
            setLoading(false)
        }
    }

    return (
        <div className="fixed inset-0 z-[100] flex items-center justify-center p-4 animate-in fade-in duration-200">
            <div className="absolute inset-0 bg-black/80 backdrop-blur-sm" onClick={onClose} />

            <div className="relative w-full max-w-md bg-[#0a050e] border border-white/10 rounded-3xl shadow-2xl overflow-hidden animate-in zoom-in-95 duration-200">
                {/* Close button */}
                <button
                    onClick={onClose}
                    className="absolute top-4 right-4 p-2 text-gray-400 hover:text-white transition-colors z-10"
                >
                    <X className="w-5 h-5" />
                </button>


                {/* Header */}
                <div className="p-8 pb-0 text-center">
                    <div className="w-12 h-12 bg-white/5 rounded-2xl flex items-center justify-center mx-auto mb-4 border border-white/10">
                        <Image src="/swiplay-logo.png" alt="Logo" width={32} height={32} className="w-8 h-8" />
                    </div>
                    <h2 className="text-2xl font-bold mb-2">{t('welcome')}</h2>
                    <p className="text-gray-400 text-sm">{t('subtitle')}</p>
                </div>

                {/* Role Switcher */}
                <div className="p-8">
                    <div className="grid grid-cols-2 gap-2 p-1 bg-white/5 rounded-xl mb-8">
                        <button
                            onClick={() => setRole('studio')}
                            className={cn(
                                "py-2.5 text-sm font-medium rounded-lg transition-all duration-300",
                                role === 'studio'
                                    ? "bg-white text-black shadow-lg"
                                    : "text-gray-400 hover:text-white"
                            )}
                        >
                            {t('studio')}
                        </button>
                        <button
                            onClick={() => setRole('creator')}
                            className={cn(
                                "py-2.5 text-sm font-medium rounded-lg transition-all duration-300",
                                role === 'creator'
                                    ? "bg-purple-500 text-white shadow-lg"
                                    : "text-gray-400 hover:text-white"
                            )}
                        >
                            {t('creator')}
                        </button>
                    </div>

                    <form onSubmit={handleSubmit} className="space-y-4">
                        <div className="space-y-2">
                            <label className="text-xs font-medium text-gray-400 ml-1">{t('email_label')}</label>
                            <div className="relative">
                                <Mail className="absolute left-4 top-3.5 w-5 h-5 text-gray-500" />
                                <input
                                    type="email"
                                    value={email}
                                    onChange={(e) => setEmail(e.target.value)}
                                    placeholder={t('email_placeholder')}
                                    className="w-full h-12 pl-12 pr-4 bg-white/5 border border-white/10 rounded-xl text-white placeholder:text-gray-600 focus:outline-none focus:border-purple-500/50 focus:ring-1 focus:ring-purple-500/50 transition-all"
                                    required
                                />
                            </div>
                        </div>

                        {message && (
                            <p className="text-xs text-red-400 text-center bg-red-500/10 py-2 rounded-lg border border-red-500/20">
                                {message}
                            </p>
                        )}

                        {/* Terms Checkbox */}
                        <div className="flex items-start space-x-2 pt-2">
                            <div className="flex h-5 items-center">
                                <input
                                    id="terms"
                                    type="checkbox"
                                    checked={acceptedTerms}
                                    onChange={(e) => setAcceptedTerms(e.target.checked)}
                                    className="h-4 w-4 rounded border-white/10 bg-white/5 text-purple-600 focus:ring-purple-500/50 focus:ring-offset-0"
                                />
                            </div>
                            <label htmlFor="terms" className="text-xs text-gray-400 leading-tight">
                                J'accepte les{' '}
                                <a href={`/${locale}/legal/terms`} target="_blank" className="text-white hover:underline">Conditions Générales d'Utilisation</a>
                            </label>
                        </div>

                        {/* Privacy Checkbox */}
                        <div className="flex items-start space-x-2">
                            <div className="flex h-5 items-center">
                                <input
                                    id="privacy"
                                    type="checkbox"
                                    checked={acceptedPrivacy}
                                    onChange={(e) => setAcceptedPrivacy(e.target.checked)}
                                    className="h-4 w-4 rounded border-white/10 bg-white/5 text-purple-600 focus:ring-purple-500/50 focus:ring-offset-0"
                                />
                            </div>
                            <label htmlFor="privacy" className="text-xs text-gray-400 leading-tight">
                                J'accepte la{' '}
                                <a href={`/${locale}/legal/privacy`} target="_blank" className="text-white hover:underline">Politique de Confidentialité</a>
                            </label>
                        </div>

                        <Button
                            type="submit"
                            disabled={loading || !acceptedTerms || !acceptedPrivacy}
                            className={cn(
                                "w-full h-12 text-base font-medium rounded-xl transition-all hover:scale-[1.02] disabled:opacity-50 disabled:hover:scale-100",
                                role === 'studio'
                                    ? "bg-white text-black hover:bg-gray-100"
                                    : "bg-purple-600 text-white hover:bg-purple-500"
                            )}
                        >
                            {loading ? (
                                <Loader2 className="w-5 h-5 animate-spin" />
                            ) : (
                                <>
                                    {t('continue')}
                                    <ArrowRight className="w-5 h-5 ml-2" />
                                </>
                            )}
                        </Button>
                    </form>
                </div>
            </div>
        </div>
    )
}
