import { useTranslations } from 'next-intl'
import { Target, Video, TrendingUp, Shield, Zap, BarChart3, CheckCircle, DollarSign } from 'lucide-react'
import { cn } from '@/lib/utils'

interface FeatureGridProps {
    activeView: 'studio' | 'creator'
}

export function FeatureGrid({ activeView }: FeatureGridProps) {
    const t = useTranslations('Landing')

    const studioFeatures = [
        {
            icon: Target,
            title: t('features.studio_feat1_title'),
            desc: t('features.studio_feat1_desc'),
            className: "md:col-span-2"
        },
        {
            icon: Shield,
            title: t('features.studio_feat2_title'),
            desc: t('features.studio_feat2_desc'),
            className: ""
        },
        {
            icon: BarChart3,
            title: t('features.studio_feat3_title'),
            desc: t('features.studio_feat3_desc'),
            className: ""
        },
        {
            icon: CheckCircle,
            title: t('features.studio_feat4_title'),
            desc: t('features.studio_feat4_desc'),
            className: "md:col-span-2"
        }
    ]

    const creatorFeatures = [
        {
            icon: DollarSign,
            title: t('features.creator_feat1_title'),
            desc: t('features.creator_feat1_desc'),
            className: "md:col-span-2"
        },
        {
            icon: Video,
            title: t('features.creator_feat2_title'),
            desc: t('features.creator_feat2_desc'),
            className: ""
        },
        {
            icon: TrendingUp,
            title: t('features.creator_feat3_title'),
            desc: t('features.creator_feat3_desc'),
            className: ""
        },
        {
            icon: Zap,
            title: t('features.creator_feat4_title'),
            desc: t('features.creator_feat4_desc'),
            className: "md:col-span-2"
        }
    ]

    const features = activeView === 'studio' ? studioFeatures : creatorFeatures

    return (
        <section className="py-24 px-6 relative z-10">
            <div className="container mx-auto max-w-6xl">
                <div className="text-center mb-16">
                    <h2 className="text-3xl md:text-5xl font-bold mb-6">
                        {t('features.title')} <span className="text-transparent bg-clip-text bg-gradient-to-r from-purple-400 to-blue-500">{t('features.title_highlight')}</span> ?
                    </h2>
                </div>

                {activeView === 'creator' ? (
                    <div className="space-y-4">
                        {/* Mobile View: Cards */}
                        <div className="md:hidden space-y-4">
                            {[0, 1, 2, 3].map((i) => (
                                <div key={i} className="bg-zinc-900/50 border border-white/10 rounded-2xl p-5 relative overflow-hidden">
                                    <div className="absolute top-0 right-0 w-20 h-20 bg-purple-500/10 blur-2xl rounded-full -mr-10 -mt-10" />

                                    <h4 className="text-gray-400 text-sm font-medium mb-4 uppercase tracking-wider">
                                        {t(`comparison.rows.${i}.criteria`)}
                                    </h4>

                                    <div className="flex justify-between items-center gap-4">
                                        <div className="flex-1">
                                            <div className="text-xs text-gray-500 mb-1">{t('comparison.headers.platforms')}</div>
                                            <div className="text-gray-300 font-medium">{t(`comparison.rows.${i}.platforms`)}</div>
                                        </div>

                                        <div className="w-px h-10 bg-white/10" />

                                        <div className="flex-1 text-right">
                                            <div className="text-xs text-purple-400 mb-1 font-bold">Swiplay</div>
                                            <div className="text-white font-bold text-lg text-shadow-purple">{t(`comparison.rows.${i}.swiplay`)}</div>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>

                        {/* Desktop View: Enhanced Table */}
                        <div className="hidden md:block glass p-1 rounded-3xl border-purple-500/20 overflow-hidden">
                            <table className="w-full text-left border-collapse">
                                <thead>
                                    <tr className="bg-white/5">
                                        <th className="p-6 text-gray-400 font-medium w-1/3">{t('comparison.headers.criteria')}</th>
                                        <th className="p-6 text-gray-400 font-medium w-1/3 text-center">{t('comparison.headers.platforms')}</th>
                                        <th className="p-6 text-purple-400 font-bold text-lg w-1/3 text-center bg-purple-500/10 border-b border-purple-500/20">
                                            {t('comparison.headers.swiplay')}
                                        </th>
                                    </tr>
                                </thead>
                                <tbody className="divide-y divide-white/5">
                                    {[0, 1, 2, 3].map((i) => (
                                        <tr key={i} className="hover:bg-white/5 transition-colors group">
                                            <td className="p-6 font-medium text-gray-300 group-hover:text-white transition-colors">
                                                {t(`comparison.rows.${i}.criteria`)}
                                            </td>
                                            <td className="p-6 text-gray-400 text-center">
                                                {t(`comparison.rows.${i}.platforms`)}
                                            </td>
                                            <td className="p-6 text-white font-bold text-lg text-center bg-purple-500/5 group-hover:bg-purple-500/10 transition-colors border-l border-purple-500/10 relative">
                                                <div className="absolute inset-0 bg-purple-500/5 opacity-0 group-hover:opacity-100 transition-opacity" />
                                                <span className="relative z-10">{t(`comparison.rows.${i}.swiplay`)}</span>
                                            </td>
                                        </tr>
                                    ))}
                                </tbody>
                            </table>
                        </div>
                    </div>
                ) : (
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                        {features.map((feature, i) => (
                            <div
                                key={i}
                                className={cn(
                                    "group relative p-8 rounded-3xl border border-white/10 bg-white/5 backdrop-blur-sm hover:bg-white/10 transition-all duration-500 hover:-translate-y-1 overflow-hidden",
                                    feature.className
                                )}
                            >
                                <div className="absolute inset-0 bg-gradient-to-br from-purple-500/10 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500" />

                                <div className="relative z-10">
                                    <div className="w-14 h-14 rounded-2xl bg-white/5 border border-white/10 flex items-center justify-center mb-6 group-hover:scale-110 transition-transform duration-500">
                                        <feature.icon className="w-7 h-7 text-purple-400" />
                                    </div>

                                    <h3 className="text-xl font-bold mb-3 text-white group-hover:text-purple-200 transition-colors">
                                        {feature.title}
                                    </h3>

                                    <p className="text-gray-400 leading-relaxed group-hover:text-gray-300 transition-colors">
                                        {feature.desc}
                                    </p>
                                </div>
                            </div>
                        ))}
                    </div>
                )}
            </div>
        </section>
    )
}
