'use client'

import { useTranslations, useLocale } from 'next-intl'
import { cn } from '@/lib/utils'
import { useState, useEffect } from 'react'
import { Link } from '@/i18n/routing'

interface HeaderProps {
    activeView?: 'studio' | 'creator'
    onViewChange?: (view: 'studio' | 'creator') => void
    showSwitcher?: boolean
}

export function Header({ activeView = 'studio', onViewChange, showSwitcher = true }: HeaderProps) {
    const t = useTranslations('Landing')
    const locale = useLocale()
    const [scrolled, setScrolled] = useState(false)
    const [adminClicks, setAdminClicks] = useState(0)

    useEffect(() => {
        const handleScroll = () => {
            setScrolled(window.scrollY > 60)
        }
        window.addEventListener('scroll', handleScroll)
        return () => window.removeEventListener('scroll', handleScroll)
    }, [])

    const handleLogoClick = () => {
        const newCount = adminClicks + 1
        setAdminClicks(newCount)
        if (newCount === 7) {
            window.location.href = `https://app.swiplay.com/${locale}/auth?admin=true`
            setAdminClicks(0)
        }
        setTimeout(() => setAdminClicks(0), 3000)
    }

    return (
        <header
            className={cn(
                "left-0 right-0 z-50 border-b transition-colors duration-300",
                "md:fixed md:top-0", // Desktop always fixed top
                scrolled ? "fixed top-0" : (showSwitcher ? "absolute top-16" : "absolute top-0"), // Mobile logic
                scrolled
                    ? "bg-black/80 backdrop-blur-xl border-white/10 py-4"
                    : "bg-transparent border-transparent py-6"
            )}
        >
            <div className="container mx-auto px-6 flex items-center justify-between relative">
                <Link href="/" className="flex items-center gap-0 cursor-pointer group z-20" onClick={handleLogoClick}>
                    <h1 className="text-3xl md:text-4xl font-bold tracking-tight flex items-center text-white">
                        <span>SWIPL</span>
                        <img
                            src="/swiplay-logo.png"
                            alt="Swiplay"
                            className="h-14 w-14 -ml-3 -mr-3 group-hover:rotate-12 transition-transform duration-500"
                        />
                        <span>Y</span>
                    </h1>
                </Link>

                {/* View Switcher - Centered (Desktop Only) */}
                {showSwitcher && onViewChange && (
                    <div className="hidden md:flex absolute left-1/2 top-1/2 -translate-x-1/2 -translate-y-1/2 p-1 bg-white/5 rounded-full border border-white/5 backdrop-blur-md z-20">
                        <button
                            onClick={() => onViewChange('studio')}
                            className={cn(
                                "px-4 py-1.5 rounded-full text-xs font-medium transition-all duration-300 cursor-pointer",
                                activeView === 'studio'
                                    ? "bg-blue-600 text-white shadow-lg shadow-blue-500/20"
                                    : "text-gray-400 hover:text-white"
                            )}
                        >
                            {t('nav.switch_studio')}
                        </button>
                        <button
                            onClick={() => onViewChange('creator')}
                            className={cn(
                                "px-4 py-1.5 rounded-full text-xs font-medium transition-all duration-300 cursor-pointer",
                                activeView === 'creator'
                                    ? "bg-purple-600 text-white shadow-lg shadow-purple-500/20"
                                    : "text-gray-400 hover:text-white"
                            )}
                        >
                            {t('nav.switch_creator')}
                        </button>
                    </div>
                )}

                <div className="flex items-center gap-4 z-20">

                    <a
                        href={`https://app.swiplay.com/${locale}/auth`}
                        className="px-6 py-2.5 rounded-full bg-white text-black font-medium text-sm hover:bg-gray-200 transition-colors cursor-pointer"
                    >
                        {t('nav.login')}
                    </a>

                    {/* Language Switcher - Desktop Only */}
                    <div className="hidden md:flex items-center gap-2 text-sm font-medium text-gray-400">
                        <Link href="/" locale="fr" className={cn("hover:text-white transition-colors", locale === 'fr' && "text-white")}>FR</Link>
                        <span className="text-white/20">|</span>
                        <Link href="/" locale="en" className={cn("hover:text-white transition-colors", locale === 'en' && "text-white")}>EN</Link>
                    </div>
                </div>
            </div>
        </header>
    )
}
