import { useTranslations, useLocale } from 'next-intl'
import { Button } from '@/components/ui/button'
import { ArrowRight, Play } from 'lucide-react'
import { cn } from '@/lib/utils'
import Prism from '@/components/ui/Prism'

interface HeroSectionProps {
    activeView: 'studio' | 'creator'
}

export function HeroSection({ activeView }: HeroSectionProps) {
    const t = useTranslations('Landing.hero')
    const locale = useLocale()

    return (
        <section className="relative min-h-screen flex items-center justify-center overflow-hidden pt-36 md:pt-20">
            {/* Prism Background */}
            <div className="absolute inset-0 z-0">
                <Prism
                    animationType="rotate"
                    timeScale={0.5}
                    height={3.5}
                    baseWidth={5.5}
                    scale={3.6}
                    hueShift={0}
                    colorFrequency={1}
                    noise={0}
                    glow={1}
                />
            </div>

            <div className="absolute inset-0 bg-black/20 z-0" />

            <div className="container mx-auto px-6 relative z-30 text-center">
                {/* Dynamic Title with Mask Effect */}
                <h1 className="text-3xl md:text-5xl lg:text-6xl font-black tracking-tight mb-6 animate-in fade-in slide-in-from-bottom-8 duration-1000 delay-200">
                    <span className="block text-white mb-2">
                        {activeView === 'studio' ? t('title_studio') : t('title_creator')}
                    </span>
                    <span className={cn(
                        "block text-transparent bg-clip-text bg-gradient-to-r",
                        activeView === 'studio'
                            ? "from-blue-400 via-indigo-400 to-purple-500"
                            : "from-purple-400 via-pink-400 to-orange-500"
                    )}>
                        {activeView === 'studio' ? t('title_studio_highlight') : t('title_creator_highlight')}
                    </span>
                </h1>

                <p className="text-lg md:text-xl text-gray-300 max-w-2xl md:max-w-4xl mx-auto mb-8 leading-relaxed animate-in fade-in slide-in-from-bottom-8 duration-1000 delay-300">
                    {activeView === 'studio'
                        ? t('subtitle_studio')
                        : t('subtitle_creator')
                    }
                </p>

                <div className="flex flex-col sm:flex-row items-center justify-center gap-4 animate-in fade-in slide-in-from-bottom-8 duration-1000 delay-500">
                    <Button
                        onClick={() => window.location.href = `https://app.swiplay.com/${locale}/auth`}
                        size="lg"
                        className={cn(
                            "h-12 px-6 text-base rounded-full text-white transition-all hover:scale-105 shadow-[0_0_40px_-10px_rgba(255,255,255,0.3)] border-0",
                            activeView === 'studio'
                                ? "bg-blue-600 hover:bg-blue-500"
                                : "bg-purple-600 hover:bg-purple-500"
                        )}
                    >
                        {activeView === 'studio' ? t('cta_studio') : t('cta_creator')}
                        <ArrowRight className="ml-2 w-4 h-4" />
                    </Button>

                    <Button
                        variant="outline"
                        size="lg"
                        onClick={() => document.getElementById('trust-section')?.scrollIntoView({ behavior: 'smooth' })}
                        className="h-12 px-6 text-base rounded-full border-white/20 bg-white/5 backdrop-blur-md hover:bg-white/10 text-white transition-all cursor-pointer"
                    >
                        <Play className="mr-2 w-4 h-4 fill-current" />
                        {t('discover')}
                    </Button>
                </div>
            </div>
        </section>
    )
}
