'use client'

import { useTranslations, useLocale } from 'next-intl'
import { useState, useEffect } from 'react'
import { useRouter, usePathname } from 'next/navigation'
import { HeroSection } from '@/components/landing/HeroSection'
import { FeatureGrid } from '@/components/landing/FeatureGrid'
import { TrustSection } from '@/components/landing/TrustSection'
import { CreatorShowcase } from '@/components/landing/CreatorShowcase'
import { AuthModal } from '@/components/landing/AuthModal'
import { HowItWorks } from '@/components/landing/HowItWorks'
import { Header } from '@/components/landing/Header'
import { Footer } from '@/components/landing/Footer'
import { cn } from '@/lib/utils'

interface LandingPageProps {
    initialView?: 'studio' | 'creator'
}

export function LandingPage({ initialView = 'creator' }: LandingPageProps) {
    const t = useTranslations('Landing')
    const locale = useLocale()
    const router = useRouter()
    const pathname = usePathname()

    const [activeView, setActiveView] = useState<'studio' | 'creator'>(initialView)
    const [showAuthModal, setShowAuthModal] = useState(false)

    // Sync state if initialView changes (though usually it won't on same route)
    useEffect(() => {
        setActiveView(initialView)
    }, [initialView])

    const handleViewChange = (view: 'studio' | 'creator') => {
        // Navigate to the clean URL
        if (view === 'studio') {
            router.push(`/${locale}/brands`)
        } else {
            router.push(`/${locale}`)
        }
    }

    return (
        <main className="min-h-screen bg-background text-foreground selection:bg-purple-500/30">
            <Header
                activeView={activeView}
                onViewChange={handleViewChange}
                showSwitcher={true}
            />

            {/* Mobile Top Switcher - Scrolls away */}
            <div className="md:hidden absolute top-2 left-1/2 -translate-x-1/2 z-40 flex p-1.5 bg-white/5 rounded-full border border-white/5 backdrop-blur-md">
                <button
                    onClick={() => handleViewChange('studio')}
                    className={cn(
                        "px-6 py-2.5 rounded-full text-sm font-medium transition-all duration-300 cursor-pointer whitespace-nowrap",
                        activeView === 'studio'
                            ? "bg-blue-600 text-white shadow-lg shadow-blue-500/20"
                            : "text-gray-400 hover:text-white"
                    )}
                >
                    {t('nav.switch_studio')}
                </button>
                <button
                    onClick={() => handleViewChange('creator')}
                    className={cn(
                        "px-6 py-2.5 rounded-full text-sm font-medium transition-all duration-300 cursor-pointer whitespace-nowrap",
                        activeView === 'creator'
                            ? "bg-purple-600 text-white shadow-lg shadow-purple-500/20"
                            : "text-gray-400 hover:text-white"
                    )}
                >
                    {t('nav.switch_creator')}
                </button>
            </div>

            <HeroSection
                activeView={activeView}
            />

            <TrustSection />

            <HowItWorks activeView={activeView} />

            <FeatureGrid activeView={activeView} />

            <CreatorShowcase />

            {/* Simple Bottom CTA */}
            <div className="py-24 px-6 text-center relative overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-b from-transparent to-purple-900/20 pointer-events-none" />
                <div className="relative z-10 max-w-3xl mx-auto">
                    <h2 className="text-3xl md:text-6xl font-black mb-6 whitespace-nowrap">
                        {t('cta_section.title')}
                    </h2>
                    <p className="text-xl text-gray-400 mb-10">
                        {t('cta_section.subtitle')}
                    </p>
                    <button
                        onClick={() => window.location.href = `https://app.swiplay.com/${locale}/auth`}
                        className="px-10 py-4 bg-white text-black rounded-full font-bold text-lg hover:scale-105 transition-transform shadow-[0_0_40px_-10px_rgba(255,255,255,0.3)]"
                    >
                        {t('cta_section.button')}
                    </button>
                </div>
            </div>

            <Footer />

            <AuthModal
                isOpen={showAuthModal}
                onClose={() => setShowAuthModal(false)}
                defaultRole={activeView}
            />
        </main>
    )
}
