'use client'

import { useLocale, useTranslations } from 'next-intl'
import { usePathname, useRouter } from '@/i18n/routing'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { useState, useTransition } from 'react'
import { toast } from 'sonner'
import { Globe } from 'lucide-react'

export function LanguageSwitcher() {
    const t = useTranslations('Settings')
    const locale = useLocale()
    const router = useRouter()
    const pathname = usePathname()
    const [isPending, startTransition] = useTransition()
    const [currentLocale, setCurrentLocale] = useState(locale)

    const onSelectChange = async (nextLocale: string) => {
        setCurrentLocale(nextLocale)

        // 1. Update user preference in DB
        try {
            await fetch('/api/user/locale', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ locale: nextLocale })
            })
        } catch (error) {
            console.error('Failed to save locale preference', error)
            // Non-blocking error, we still switch UI
        }

        // 2. Switch URL via next-intl
        startTransition(() => {
            router.replace(pathname, { locale: nextLocale })
            toast.success(t('saved'))
        })
    }

    return (
        <div className="flex items-center justify-between p-4 bg-white/5 rounded-xl border border-white/10">
            <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-full bg-purple-500/20 flex items-center justify-center text-purple-400">
                    <Globe className="w-5 h-5" />
                </div>
                <div>
                    <h3 className="font-medium text-white">{t('language')}</h3>
                    <p className="text-sm text-gray-400">{t('language_desc')}</p>
                </div>
            </div>

            <Select value={currentLocale} onValueChange={onSelectChange} disabled={isPending}>
                <SelectTrigger className="w-[140px] bg-black/20 border-white/10">
                    <SelectValue placeholder="Language" />
                </SelectTrigger>
                <SelectContent>
                    <SelectItem value="fr">Français 🇫🇷</SelectItem>
                    <SelectItem value="en">English 🇺🇸</SelectItem>
                </SelectContent>
            </Select>
        </div>
    )
}
