'use client'

import { useState } from 'react'
import { useRouter } from 'next/navigation'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { Card, CardContent } from '@/components/ui/card'
import { campaignsApi } from '@/lib/api'
import { toast } from 'sonner'
import { ArrowLeft, ArrowRight, Check, Upload, Gamepad2, Coins, FileText, Image as ImageIcon } from 'lucide-react'
import { ImageUpload } from '@/components/ImageUpload'
import { AssetUpload, Asset } from '@/components/AssetUpload'
import { RichTextEditor } from '@/components/ui/RichTextEditor'
import { Textarea } from '@/components/ui/textarea'

const STEPS = [
    { id: 'details', title: 'Détails', icon: FileText },
    { id: 'visuals', title: 'Visuels', icon: ImageIcon },
    { id: 'budget', title: 'Budget', icon: Coins },
    { id: 'keys', title: 'Clés', icon: Gamepad2 },
    { id: 'assets', title: 'Assets', icon: Upload },
    { id: 'review', title: 'Récapitulatif', icon: Check },
]

export function CampaignWizard() {
    const router = useRouter()
    const [currentStep, setCurrentStep] = useState(0)
    const [loading, setLoading] = useState(false)

    const [formData, setFormData] = useState({
        title: '',
        description: '',
        tags: [] as string[],
        coverImage: '',
        icon: '',
        images: [] as string[],
        videos: [] as string[],
        budget: '',
        pricePerView: '',
        creatorCap: '',
        startDate: '',
        endDate: '',
        platform: 'tiktok',
        gameKeys: [] as { platform: string, keys: string }[],
        assets: [] as Asset[]
    })

    const handleNext = () => {
        if (currentStep < STEPS.length - 1) {
            setCurrentStep(prev => prev + 1)
        } else {
            handleSubmit()
        }
    }

    const handleBack = () => {
        if (currentStep > 0) {
            setCurrentStep(prev => prev - 1)
        }
    }

    const handleSubmit = async () => {
        setLoading(true)
        try {
            const { data, error } = await campaignsApi.create({
                ...formData,
                budget: parseFloat(formData.budget),
                pricePerView: parseFloat(formData.pricePerView),
                creatorCap: formData.creatorCap ? parseFloat(formData.creatorCap) : undefined,
                gameKeys: formData.gameKeys,
                assets: formData.assets.map(a => ({
                    name: a.name,
                    type: a.type,
                    url: a.url || '',
                    size: a.size
                }))
            })

            if (error) throw new Error(error)

            if (data) {
                toast.success('Campagne créée ! Redirection vers le paiement...')
                router.push(`/payment?campaignId=${data.campaign.id}&amount=${Math.round(parseFloat(formData.budget) * 100)}`)
            }
        } catch (error: any) {
            toast.error(error.message || 'Erreur lors de la création')
        } finally {
            setLoading(false)
        }
    }

    const updateField = (field: string, value: any) => {
        setFormData(prev => ({ ...prev, [field]: value }))
    }

    const renderStepContent = () => {
        switch (currentStep) {
            case 0: // Details
                return (
                    <div className="space-y-6">
                        <div className="space-y-2">
                            <Label>Titre de la campagne *</Label>
                            <Input
                                value={formData.title}
                                onChange={(e) => updateField('title', e.target.value)}
                                placeholder="Ex: Lancement Saison 2"
                            />
                        </div>
                        <div className="space-y-2">
                            <Label>Tags du jeu</Label>
                            <div className="flex flex-wrap gap-2 p-3 bg-white/5 rounded-lg border border-white/10 min-h-[80px]">
                                {[
                                    { value: 'solo', label: 'Solo' },
                                    { value: 'multiplayer', label: 'Multijoueur' },
                                    { value: 'coop', label: 'Coopératif' },
                                    { value: 'pvp', label: 'PvP' },
                                    { value: 'action', label: 'Action' },
                                    { value: 'adventure', label: 'Aventure' },
                                    { value: 'rpg', label: 'RPG' },
                                    { value: 'fps', label: 'FPS' },
                                    { value: 'horror', label: 'Horreur' },
                                    { value: 'survival', label: 'Survie' },
                                    { value: 'roguelike', label: 'Roguelike' },
                                    { value: 'puzzle', label: 'Puzzle' },
                                    { value: 'platformer', label: 'Plateformes' },
                                    { value: 'racing', label: 'Course' },
                                    { value: 'simulation', label: 'Simulation' },
                                    { value: 'strategy', label: 'Stratégie' },
                                    { value: 'openworld', label: 'Monde Ouvert' },
                                    { value: 'indie', label: 'Indé' },
                                    { value: 'casual', label: 'Casual' },
                                ].map((tag) => (
                                    <button
                                        key={tag.value}
                                        type="button"
                                        onClick={() => {
                                            const currentTags = formData.tags
                                            if (currentTags.includes(tag.value)) {
                                                updateField('tags', currentTags.filter((t: string) => t !== tag.value))
                                            } else {
                                                updateField('tags', [...currentTags, tag.value])
                                            }
                                        }}
                                        className={`px-3 py-1.5 rounded-full text-sm font-medium transition-all ${formData.tags.includes(tag.value)
                                                ? 'bg-purple-600 text-white'
                                                : 'bg-white/10 text-gray-300 hover:bg-white/20'
                                            }`}
                                    >
                                        {tag.label}
                                    </button>
                                ))}
                            </div>
                            <p className="text-xs text-gray-500">Cliquez pour sélectionner les tags correspondant à votre jeu</p>
                        </div>
                        <div className="space-y-2">
                            <Label>Description *</Label>
                            <RichTextEditor
                                content={formData.description}
                                onChange={(content) => updateField('description', content)}
                                placeholder="Décrivez votre campagne..."
                            />
                        </div>
                        <div className="grid grid-cols-2 gap-4">
                            <div className="grid grid-cols-2 gap-4">
                                <div>
                                    <Label>Date de début</Label>
                                    <Input type="date" value={formData.startDate} onChange={(e) => updateField('startDate', e.target.value)} />
                                </div>
                                <div>
                                    <Label>Date de fin (Optionnel)</Label>
                                    <Input type="date" value={formData.endDate} onChange={(e) => updateField('endDate', e.target.value)} />
                                    <p className="text-xs text-gray-500 mt-1">Laisser vide pour une durée indéterminée (jusqu'à épuisement du budget)</p>
                                </div>
                            </div>
                        </div>
                    </div>
                )
            case 1: // Visuals
                return (
                    <div className="space-y-6">
                        <div className="grid md:grid-cols-2 gap-6">
                            <div className="space-y-2">
                                <Label>Image de couverture (16:9)</Label>
                                {formData.coverImage ? (
                                    <div className="relative h-48 rounded-lg overflow-hidden group">
                                        <img src={formData.coverImage} className="w-full h-full object-cover" />
                                        <Button
                                            variant="destructive"
                                            size="sm"
                                            className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity"
                                            onClick={() => updateField('coverImage', '')}
                                        >
                                            Supprimer
                                        </Button>
                                    </div>
                                ) : (
                                    <ImageUpload onUploadComplete={(path) => updateField('coverImage', path)} label="Cover Image" />
                                )}
                            </div>
                            <div className="space-y-2">
                                <Label>Icône (Carré)</Label>
                                {formData.icon ? (
                                    <div className="relative h-32 w-32 rounded-lg overflow-hidden group">
                                        <img src={formData.icon} className="w-full h-full object-cover" />
                                        <Button
                                            variant="destructive"
                                            size="sm"
                                            className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity"
                                            onClick={() => updateField('icon', '')}
                                        >
                                            Supprimer
                                        </Button>
                                    </div>
                                ) : (
                                    <ImageUpload onUploadComplete={(path) => updateField('icon', path)} label="Icon" className="w-32" />
                                )}
                            </div>
                        </div>

                        <div className="space-y-2">
                            <Label>Galerie d'Images (1-10)</Label>
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                                {formData.images.map((img, idx) => (
                                    <div key={idx} className="relative aspect-video rounded-lg overflow-hidden group bg-zinc-800">
                                        <img src={img} className="w-full h-full object-cover" />
                                        <Button
                                            variant="destructive"
                                            size="icon"
                                            className="absolute top-1 right-1 h-6 w-6 opacity-0 group-hover:opacity-100 transition-opacity"
                                            onClick={() => updateField('images', formData.images.filter((_, i) => i !== idx))}
                                        >
                                            <span className="sr-only">Supprimer</span>
                                            ×
                                        </Button>
                                    </div>
                                ))}
                                {formData.images.length < 10 && (
                                    <ImageUpload
                                        onUploadComplete={(path) => updateField('images', [...formData.images, path])}
                                        label="Ajouter"
                                        className="h-full min-h-[100px]"
                                    />
                                )}
                            </div>
                        </div>

                        <div className="space-y-2">
                            <Label>Galerie Vidéos (1-3) (Liens YouTube/MP4)</Label>
                            <div className="space-y-2">
                                {formData.videos.map((video, idx) => (
                                    <div key={idx} className="flex gap-2">
                                        <Input value={video} readOnly className="bg-zinc-900" />
                                        <Button variant="destructive" onClick={() => updateField('videos', formData.videos.filter((_, i) => i !== idx))}>Supprimer</Button>
                                    </div>
                                ))}
                                {formData.videos.length < 3 && (
                                    <div className="flex gap-2">
                                        <Input
                                            placeholder="URL de la vidéo (YouTube, MP4...)"
                                            onKeyDown={(e) => {
                                                if (e.key === 'Enter') {
                                                    const val = (e.target as HTMLInputElement).value
                                                    if (val) {
                                                        updateField('videos', [...formData.videos, val])
                                                            ; (e.target as HTMLInputElement).value = ''
                                                    }
                                                }
                                            }}
                                        />
                                        <Button variant="secondary" onClick={(e) => {
                                            const input = e.currentTarget.previousElementSibling as HTMLInputElement
                                            if (input.value) {
                                                updateField('videos', [...formData.videos, input.value])
                                                input.value = ''
                                            }
                                        }}>Ajouter</Button>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                )
            case 2: // Budget
                return (
                    <div className="space-y-6">
                        <div className="grid md:grid-cols-2 gap-6">
                            <div className="space-y-2">
                                <Label>Budget Total (€) *</Label>
                                <Input
                                    type="number"
                                    value={formData.budget}
                                    onChange={(e) => updateField('budget', e.target.value)}
                                    placeholder="1000"
                                />
                            </div>
                            <div className="space-y-2">
                                <Label>Prix par Vue (€) *</Label>
                                <Input
                                    type="number"
                                    step="0.001"
                                    value={formData.pricePerView}
                                    onChange={(e) => updateField('pricePerView', e.target.value)}
                                    placeholder="0.02"
                                />
                            </div>
                            <div className="space-y-2">
                                <Label>Plafond par Créateur (€) (Optionnel)</Label>
                                <Input
                                    type="number"
                                    value={formData.creatorCap}
                                    onChange={(e) => updateField('creatorCap', e.target.value)}
                                    placeholder="Ex: 500"
                                />
                                <p className="text-xs text-gray-400">Limite maximale qu'un seul créateur peut gagner.</p>
                            </div>
                        </div>
                    </div>
                )
            case 3: // Game Keys
                return (
                    <div className="space-y-6">
                        <div className="space-y-4">
                            <Label>Clés de jeu (Une par ligne)</Label>
                            <div className="grid gap-4">
                                <div className="space-y-2">
                                    <Label className="text-xs text-gray-400">PC / Steam</Label>
                                    <Textarea
                                        placeholder="XXXX-XXXX-XXXX..."
                                        rows={5}
                                        className="font-mono text-sm bg-zinc-950/50"
                                        onChange={(e) => {
                                            const keys = e.target.value
                                            const existing = formData.gameKeys.filter(k => k.platform !== 'PC')
                                            if (keys.trim()) {
                                                updateField('gameKeys', [...existing, { platform: 'PC', keys }])
                                            } else {
                                                updateField('gameKeys', existing)
                                            }
                                        }}
                                    />
                                </div>
                            </div>
                            <p className="text-xs text-gray-400">Ces clés seront distribuées automatiquement aux créateurs approuvés.</p>
                        </div>
                    </div>
                )
            case 4: // Assets
                return (
                    <div className="space-y-6">
                        <Label>Assets de campagne (Images, Vidéos, PDF)</Label>
                        <AssetUpload
                            value={formData.assets}
                            onChange={(assets) => updateField('assets', assets)}
                        />
                    </div>
                )
            case 5: // Review
                return (
                    <div className="space-y-6">
                        <h3 className="text-xl font-bold">Récapitulatif</h3>
                        <div className="grid gap-4 text-sm">
                            <div className="flex justify-between border-b border-white/5 pb-2">
                                <span className="text-gray-400">Campagne</span>
                                <span>{formData.title}</span>
                            </div>
                            <div className="flex justify-between border-b border-white/5 pb-2">
                                <span className="text-gray-400">Budget</span>
                                <span className="font-bold text-green-400">{formData.budget} €</span>
                            </div>
                            <div className="flex justify-between border-b border-white/5 pb-2">
                                <span className="text-gray-400">Prix / Vue</span>
                                <span>{formData.pricePerView} €</span>
                            </div>
                            <div className="flex justify-between border-b border-white/5 pb-2">
                                <span className="text-gray-400">Clés de jeu</span>
                                <span>{formData.gameKeys.reduce((acc, g) => acc + g.keys.split('\n').filter(k => k.trim()).length, 0)} clés</span>
                            </div>
                            <div className="flex justify-between border-b border-white/5 pb-2">
                                <span className="text-gray-400">Assets</span>
                                <span>{formData.assets.length} fichiers</span>
                            </div>
                        </div>
                    </div>
                )
            default:
                return null
        }
    }

    return (
        <div className="max-w-4xl mx-auto">
            {/* Stepper */}
            <div className="flex justify-between mb-8 relative">
                <div className="absolute top-1/2 left-0 right-0 h-0.5 bg-zinc-800 -z-10" />
                {STEPS.map((step, idx) => {
                    const Icon = step.icon
                    const isActive = idx === currentStep
                    const isCompleted = idx < currentStep
                    return (
                        <div key={step.id} className="flex flex-col items-center gap-2 bg-zinc-950 px-2">
                            <div className={`w-10 h-10 rounded-full flex items-center justify-center border-2 transition-colors ${isActive ? 'border-purple-500 bg-purple-500/10 text-purple-500' : isCompleted ? 'border-green-500 bg-green-500/10 text-green-500' : 'border-zinc-700 bg-zinc-900 text-gray-500'}`}>
                                {isCompleted ? <Check className="w-5 h-5" /> : <Icon className="w-5 h-5" />}
                            </div>
                            <span className={`text-xs font-medium ${isActive ? 'text-white' : 'text-gray-500'}`}>{step.title}</span>
                        </div>
                    )
                })}
            </div>

            <Card className="bg-zinc-900/40 border-zinc-800">
                <CardContent className="p-6">
                    {renderStepContent()}

                    <div className="flex justify-between mt-8 pt-4 border-t border-white/5">
                        <Button
                            variant="ghost"
                            onClick={handleBack}
                            disabled={currentStep === 0 || loading}
                            className="text-gray-400 hover:text-white"
                        >
                            <ArrowLeft className="w-4 h-4 mr-2" /> Retour
                        </Button>

                        <Button
                            onClick={handleNext}
                            disabled={loading}
                            className="bg-purple-600 hover:bg-purple-700"
                        >
                            {currentStep === STEPS.length - 1 ? (
                                loading ? 'Création...' : 'Créer et Payer'
                            ) : (
                                <>Suivant <ArrowRight className="w-4 h-4 ml-2" /></>
                            )}
                        </Button>
                    </div>
                </CardContent>
            </Card>
        </div>
    )
}
