import { useEffect, useState, useCallback } from 'react'
import { useAuth } from '@/lib/auth-store'

interface UseBalanceOptions {
    pollingInterval?: number // in milliseconds, default 60000 (1 minute)
    enabled?: boolean
}

/**
 * Hook for auto-refreshing user balance
 * Polls the API at regular intervals to keep balance updated
 */
export function useBalance(options: UseBalanceOptions = {}) {
    const { pollingInterval = 60000, enabled = true } = options
    const token = useAuth((state) => state.accessToken)

    const [balance, setBalance] = useState<number>(0)
    const [loading, setLoading] = useState(false)
    const [error, setError] = useState<string | null>(null)

    const fetchBalance = useCallback(async () => {
        if (!token) return

        try {
            setLoading(true)
            const response = await fetch('/api/creator/balance', {
                headers: { 'Authorization': `Bearer ${token}` }
            })

            if (!response.ok) throw new Error('Failed to fetch balance')

            const data = await response.json()
            setBalance(data.balance)

            setError(null)
        } catch (err: any) {
            setError(err.message)
        } finally {
            setLoading(false)
        }
    }, [token])

    // Initial fetch
    useEffect(() => {
        if (enabled && token) {
            fetchBalance()
        }
    }, [enabled, token, fetchBalance])

    // Polling
    useEffect(() => {
        if (!enabled || !token || pollingInterval <= 0) return

        const intervalId = setInterval(fetchBalance, pollingInterval)

        return () => clearInterval(intervalId)
    }, [enabled, token, pollingInterval, fetchBalance])

    return {
        balance,
        loading,
        error,
        refresh: fetchBalance
    }
}
