import { NextRequest } from 'next/server'
import { db } from '@/lib/db'
import { users } from '@/lib/db/schema'
import { eq } from 'drizzle-orm'
import jwt from 'jsonwebtoken'

interface JWTPayload {
  userId: string
  email: string
  role: string
}

interface AuthResult {
  success: boolean
  userId?: string
  email?: string
  role?: string
  error?: string
}

export async function authenticateRequest(request: NextRequest): Promise<AuthResult> {
  const authHeader = request.headers.get('authorization')

  if (!authHeader) {
    return { success: false, error: 'Non autorisé - Token requis' }
  }

  const token = authHeader.replace('Bearer ', '')

  try {
    // Verify JWT token
    const decoded = jwt.verify(token, process.env.JWT_SECRET!) as JWTPayload

    const user = await db.query.users.findFirst({
      where: eq(users.id, decoded.userId)
    })

    if (!user) {
      return { success: false, error: 'Utilisateur introuvable' }
    }

    return {
      success: true,
      userId: user.id,
      email: user.email,
      role: user.role
    }
  } catch (error) {
    if (error instanceof jwt.TokenExpiredError) {
      return { success: false, error: 'Token expiré' }
    }

    return { success: false, error: 'Token invalide' }
  }
}

export function requireRole(userRole: string | undefined, requiredRole: string): { allowed: boolean; error?: string } {
  if (!userRole || userRole !== requiredRole) {
    return { allowed: false, error: `Accès interdit - ${requiredRole} requis` }
  }
  return { allowed: true }
}

export async function requireAuth(request: NextRequest) {
  const authHeader = request.headers.get('authorization')

  if (!authHeader) {
    return {
      user: null,
      response: new Response(JSON.stringify({ error: 'Non autorisé - Token requis' }), {
        status: 401,
        headers: { 'Content-Type': 'application/json' }
      })
    }
  }

  const token = authHeader.replace('Bearer ', '')

  try {
    // Verify JWT token
    const decoded = jwt.verify(token, process.env.JWT_SECRET!) as JWTPayload

    const user = await db.query.users.findFirst({
      where: eq(users.id, decoded.userId)
    })

    if (!user) {
      return {
        user: null,
        response: new Response(JSON.stringify({ error: 'Utilisateur introuvable' }), {
          status: 404,
          headers: { 'Content-Type': 'application/json' }
        })
      }
    }

    return { user, response: null }
  } catch (error) {
    if (error instanceof jwt.TokenExpiredError) {
      return {
        user: null,
        response: new Response(JSON.stringify({ error: 'Token expiré' }), {
          status: 401,
          headers: { 'Content-Type': 'application/json' }
        })
      }
    }

    return {
      user: null,
      response: new Response(JSON.stringify({ error: 'Token invalide' }), {
        status: 401,
        headers: { 'Content-Type': 'application/json' }
      })
    }
  }
}

export async function requireAdmin(request: NextRequest) {
  const authResult = await requireAuth(request)

  if (authResult.response) {
    return authResult
  }

  if (authResult.user?.role !== 'admin') {
    return {
      user: null,
      response: new Response(JSON.stringify({ error: 'Accès interdit - Administrateur requis' }), {
        status: 403,
        headers: { 'Content-Type': 'application/json' }
      })
    }
  }

  return authResult
}
